#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
새로운 A/S 확인서 시스템 종합 테스트

실제 세원 A/S 확인서 기반 시스템의 모든 기능을 테스트합니다.
"""

import sys
from pathlib import Path
from datetime import datetime

# 프로젝트 루트를 Python 경로에 추가
project_root = Path(__file__).parent
sys.path.insert(0, str(project_root))

from src.n1vi.models.equipment import Equipment
from src.n1vi.models.as_report import AsReportData, AsReportSettings
from src.n1vi.services.equipment_service import EquipmentService
from src.n1vi.services.as_service import AsService


def test_equipment_service():
    """설비 서비스 테스트"""
    print("1️⃣ 설비 서비스 테스트")
    print("-" * 50)
    
    try:
        service = EquipmentService()
        
        # 데이터 로드 테스트
        success = service.load_data()
        if success:
            equipment_list = service.get_all_equipment()
            print(f"✅ 데이터 로드 성공: {len(equipment_list)}개 설비")
            
            if equipment_list:
                sample = equipment_list[0]
                print(f"✅ 샘플 설비: {sample.company_name} - {sample.machine_number}")
        else:
            print("❌ 데이터 로드 실패")
            return False
        
        # 검색 테스트
        search_results = service.search_equipment("삼성")
        print(f"✅ 검색 테스트 ('삼성'): {len(search_results)}건 발견")
        
        return True
        
    except Exception as e:
        print(f"❌ 설비 서비스 테스트 실패: {e}")
        return False


def test_as_models():
    """A/S 모델 테스트"""
    print("\n2️⃣ A/S 데이터 모델 테스트")
    print("-" * 50)
    
    try:
        # AsReportData 테스트
        as_data = AsReportData(
            company_name="테스트 회사",
            model_name="TEST-100",
            serial_number="12345",
            receipt_date="2025.07.08",
            confirmer="김테스터"
        )
        
        # 파일명 생성 테스트
        filename = as_data.get_filename()
        print(f"✅ 파일명 생성: {filename}")
        
        # 필수 필드 검증 테스트
        is_valid, errors = as_data.validate_required_fields()
        if is_valid:
            print("✅ 필수 필드 검증 통과")
        else:
            print(f"❌ 필수 필드 검증 실패: {errors}")
        
        # 셀 매핑 테스트
        cell_mapping = as_data.get_cell_mapping()
        print(f"✅ 셀 매핑 생성: {len(cell_mapping)}개 셀")
        
        # 설정 테스트
        settings = AsReportSettings()
        print(f"✅ 기본 담당자: {settings.default_manager}")
        
        return True
        
    except Exception as e:
        print(f"❌ A/S 모델 테스트 실패: {e}")
        return False


def test_as_service():
    """A/S 서비스 테스트"""
    print("\n3️⃣ A/S 서비스 테스트")
    print("-" * 50)
    
    try:
        service = AsService()
        
        # 템플릿 검증 테스트
        if service.validate_template():
            print("✅ 템플릿 파일 검증 통과")
        else:
            print("❌ 템플릿 파일 검증 실패")
            return False
        
        # 샘플 설비 데이터로 A/S 데이터 생성 테스트
        sample_equipment = Equipment(
            company_name="테스트 회사",
            machine_number="테스트기",
            model_name="TEST-200",
            serial_number="67890",
            manufacturing_date="2024-01-01"
        )
        
        as_data = service.create_equipment_data(sample_equipment)
        print(f"✅ A/S 데이터 생성: {as_data.company_name} - {as_data.model_name}")
        
        # 기본 담당자 테스트
        default_manager = service.get_default_manager()
        print(f"✅ 기본 담당자: {default_manager}")
        
        return True
        
    except Exception as e:
        print(f"❌ A/S 서비스 테스트 실패: {e}")
        return False


def test_template_file():
    """템플릿 파일 테스트"""
    print("\n4️⃣ 템플릿 파일 테스트")
    print("-" * 50)
    
    try:
        template_path = project_root / "세원_SHT-300VX_1호기_AS 확인서_250708.xlsx"
        
        if template_path.exists():
            size_mb = template_path.stat().st_size / (1024 * 1024)
            print(f"✅ 템플릿 파일 발견: {template_path.name} ({size_mb:.2f}MB)")
            
            # openpyxl로 파일 읽기 테스트
            from openpyxl import load_workbook
            
            workbook = load_workbook(template_path, data_only=True)
            worksheet = workbook.active
            
            print(f"✅ 엑셀 파일 열기 성공: {worksheet.title}")
            print(f"✅ 크기: {worksheet.max_row}행 x {worksheet.max_column}열")
            
            # 주요 셀 확인
            test_cells = ['G4', 'AA4', 'N5', 'AA5']
            for cell_addr in test_cells:
                value = worksheet[cell_addr].value
                print(f"✅ {cell_addr}: {value}")
            
            workbook.close()
            return True
        else:
            print(f"❌ 템플릿 파일 없음: {template_path}")
            return False
        
    except Exception as e:
        print(f"❌ 템플릿 파일 테스트 실패: {e}")
        return False


def test_file_system():
    """파일 시스템 테스트"""
    print("\n5️⃣ 파일 시스템 테스트")
    print("-" * 50)
    
    try:
        # 데이터 파일 확인
        data_file = project_root / "data" / "equipment_data.csv"
        if data_file.exists():
            size_kb = data_file.stat().st_size / 1024
            print(f"✅ 데이터 파일: {data_file.name} ({size_kb:.1f}KB)")
        else:
            print(f"❌ 데이터 파일 없음: {data_file}")
            return False
        
        # 출력 폴더 확인/생성
        output_dir = project_root / "output"
        output_dir.mkdir(exist_ok=True)
        print(f"✅ 출력 폴더: {output_dir}")
        
        # 스크립트 폴더 확인
        scripts_dir = project_root / "scripts"
        if scripts_dir.exists():
            scripts = list(scripts_dir.glob("*.py"))
            print(f"✅ 스크립트 폴더: {len(scripts)}개 스크립트")
        
        return True
        
    except Exception as e:
        print(f"❌ 파일 시스템 테스트 실패: {e}")
        return False


def test_integration():
    """통합 테스트 (실제 A/S 확인서 생성)"""
    print("\n6️⃣ 통합 테스트 (A/S 확인서 생성)")
    print("-" * 50)
    
    try:
        # 서비스 초기화
        equipment_service = EquipmentService()
        as_service = AsService()
        
        # 데이터 로드
        if not equipment_service.load_data():
            print("❌ 설비 데이터 로드 실패")
            return False
        
        # 첫 번째 설비로 테스트
        all_equipment = equipment_service.get_all_equipment()
        if not all_equipment:
            print("❌ 설비 데이터가 없습니다")
            return False
        
        test_equipment = all_equipment[0]
        print(f"✅ 테스트 설비: {test_equipment.company_name} - {test_equipment.machine_number}")
        
        # A/S 데이터 생성
        as_data = as_service.create_equipment_data(test_equipment)
        
        # 테스트 정보 입력
        as_data.receipt_date = datetime.now().strftime("%Y.%m.%d")
        as_data.as_request = "테스트 A/S 신청"
        as_data.as_repair = "테스트 수리 완료"
        as_data.check_items = "테스트 체크 완료"
        as_data.confirmer = "테스트 확인자"
        
        # A/S 확인서 생성
        success, file_path, error_msg = as_service.generate_as_report(as_data)
        
        if success:
            output_file = Path(file_path)
            if output_file.exists():
                size_kb = output_file.stat().st_size / 1024
                print(f"✅ A/S 확인서 생성 성공: {output_file.name} ({size_kb:.1f}KB)")
                
                # 파일 내용 검증
                from openpyxl import load_workbook
                workbook = load_workbook(file_path, data_only=True)
                worksheet = workbook.active
                
                # 몇 개 셀 값 확인
                g4_value = worksheet['G4'].value  # 업체명
                n5_value = worksheet['N5'].value  # 모델명
                g8_value = worksheet['G8'].value  # 접수일
                
                print(f"✅ 생성된 파일 검증:")
                print(f"   G4 (업체명): {g4_value}")
                print(f"   N5 (모델명): {n5_value}")
                print(f"   G8 (접수일): {g8_value}")
                
                workbook.close()
                return True
            else:
                print(f"❌ 생성된 파일을 찾을 수 없음: {file_path}")
                return False
        else:
            print(f"❌ A/S 확인서 생성 실패: {error_msg}")
            return False
        
    except Exception as e:
        print(f"❌ 통합 테스트 실패: {e}")
        return False


def main():
    """메인 테스트 함수"""
    print("🔧 N1VI 새로운 A/S 확인서 시스템 종합 테스트")
    print("=" * 70)
    print(f"⏰ 테스트 시작: {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    print()
    
    # 테스트 실행
    tests = [
        ("파일 시스템", test_file_system),
        ("설비 서비스", test_equipment_service),
        ("A/S 모델", test_as_models),
        ("A/S 서비스", test_as_service),
        ("템플릿 파일", test_template_file),
        ("통합 테스트", test_integration),
    ]
    
    passed = 0
    failed = 0
    
    for test_name, test_func in tests:
        try:
            if test_func():
                passed += 1
                print(f"✅ {test_name} 테스트 통과")
            else:
                failed += 1
                print(f"❌ {test_name} 테스트 실패")
        except Exception as e:
            failed += 1
            print(f"❌ {test_name} 테스트 예외: {e}")
        
        print()
    
    # 결과 요약
    print("=" * 70)
    print("📊 테스트 결과 요약")
    print("-" * 30)
    print(f"✅ 통과: {passed}개")
    print(f"❌ 실패: {failed}개")
    print(f"📈 성공률: {passed/(passed+failed)*100:.1f}%")
    
    if failed == 0:
        print("\n🎉 모든 테스트가 성공적으로 완료되었습니다!")
        print("🚀 시스템이 정상적으로 작동할 준비가 되었습니다.")
    else:
        print(f"\n⚠️ {failed}개의 테스트가 실패했습니다.")
        print("🔧 문제를 해결한 후 다시 시도해주세요.")
    
    return failed == 0


if __name__ == "__main__":
    success = main()
    sys.exit(0 if success else 1) 