#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
기술 스택 선택 UI
개발자가 프로젝트에 사용할 백엔드/프론트엔드 기술 조합을 선택할 수 있는 UI
"""

import sys
import json
import os
from datetime import datetime
from PySide6.QtWidgets import (
    QApplication, QMainWindow, QWidget, QVBoxLayout, QHBoxLayout,
    QLabel, QRadioButton, QButtonGroup, QPushButton, QTextEdit,
    QGroupBox, QScrollArea, QMessageBox, QFrame
)
from PySide6.QtCore import Qt, QSize
from PySide6.QtGui import QFont, QPixmap, QIcon

class TechStackSelector(QMainWindow):
    def __init__(self):
        super().__init__()
        self.selected_stack = None
        self.init_ui()
        
    def init_ui(self):
        """UI 초기화"""
        self.setWindowTitle("N1VI 프로젝트 - 기술 스택 선택")
        self.setMinimumSize(800, 700)
        self.setStyleSheet("""
            QMainWindow {
                background-color: #f5f5f5;
            }
            QGroupBox {
                font-weight: bold;
                border: 2px solid #cccccc;
                border-radius: 10px;
                margin-top: 1ex;
                padding-top: 10px;
            }
            QGroupBox::title {
                subcontrol-origin: margin;
                left: 10px;
                padding: 0 10px 0 10px;
                color: #333333;
            }
            QRadioButton {
                font-size: 14px;
                font-weight: bold;
                padding: 5px;
            }
            QRadioButton::indicator {
                width: 18px;
                height: 18px;
            }
            QPushButton {
                font-size: 14px;
                font-weight: bold;
                padding: 10px 20px;
                border-radius: 5px;
                background-color: #4CAF50;
                color: white;
                border: none;
            }
            QPushButton:hover {
                background-color: #45a049;
            }
            QPushButton:pressed {
                background-color: #3d8b40;
            }
            QTextEdit {
                border: 1px solid #cccccc;
                border-radius: 5px;
                padding: 10px;
                background-color: white;
                font-family: "Segoe UI", Arial, sans-serif;
            }
        """)
        
        # 중앙 위젯 생성
        central_widget = QWidget()
        self.setCentralWidget(central_widget)
        
        # 메인 레이아웃
        main_layout = QVBoxLayout(central_widget)
        main_layout.setSpacing(20)
        main_layout.setContentsMargins(30, 30, 30, 30)
        
        # 제목
        title_label = QLabel("🚀 N1VI 프로젝트 기술 스택 선택")
        title_label.setFont(QFont("Arial", 18, QFont.Bold))
        title_label.setStyleSheet("color: #2c3e50; margin-bottom: 10px;")
        title_label.setAlignment(Qt.AlignCenter)
        main_layout.addWidget(title_label)
        
        # 설명
        desc_label = QLabel("설비 정보 조회 및 보고서 생성 애플리케이션을 위한 기술 스택을 선택해주세요.")
        desc_label.setFont(QFont("Arial", 12))
        desc_label.setStyleSheet("color: #555555; margin-bottom: 20px;")
        desc_label.setAlignment(Qt.AlignCenter)
        main_layout.addWidget(desc_label)
        
        # 스크롤 영역
        scroll_area = QScrollArea()
        scroll_widget = QWidget()
        scroll_layout = QVBoxLayout(scroll_widget)
        
        # 기술 스택 옵션들
        self.tech_stacks = self.get_tech_stack_options()
        self.button_group = QButtonGroup()
        
        for i, stack in enumerate(self.tech_stacks):
            stack_widget = self.create_stack_option(stack, i)
            scroll_layout.addWidget(stack_widget)
            
        scroll_area.setWidget(scroll_widget)
        scroll_area.setWidgetResizable(True)
        main_layout.addWidget(scroll_area)
        
        # 선택 버튼
        button_layout = QHBoxLayout()
        self.select_button = QPushButton("선택된 기술 스택으로 진행")
        self.select_button.setEnabled(False)
        self.select_button.clicked.connect(self.confirm_selection)
        button_layout.addStretch()
        button_layout.addWidget(self.select_button)
        button_layout.addStretch()
        main_layout.addLayout(button_layout)
        
    def get_tech_stack_options(self):
        """기술 스택 옵션 정보 반환"""
        return [
            {
                "name": "Python (PySide6)",
                "subtitle": "데스크탑 네이티브 애플리케이션 [권장]",
                "description": "Python 기반 데스크탑 애플리케이션으로 빠른 개발과 안정성을 제공합니다.",
                "advantages": [
                    "빠른 개발 속도 및 풍부한 라이브러리 생태계",
                    "pandas, openpyxl 등 데이터 처리에 최적화",
                    "네이티브 성능과 시스템 리소스 효율성",
                    "단일 실행 파일로 배포 가능 (PyInstaller)",
                    "오프라인 환경에서 완벽한 동작"
                ],
                "disadvantages": [
                    "UI 디자인의 제한적 유연성",
                    "웹 기반 UI 대비 현대적 디자인 어려움",
                    "크로스 플랫폼 배포 시 테스트 필요"
                ],
                "tech_details": "PySide6 + SQLite + pandas + openpyxl + requests",
                "recommended": True
            },
            {
                "name": "Electron",
                "subtitle": "웹 기술 기반 데스크탑 애플리케이션",
                "description": "HTML, CSS, JavaScript를 사용하여 현대적인 UI를 가진 데스크탑 앱을 개발합니다.",
                "advantages": [
                    "웹 기술 사용으로 현대적이고 유연한 UI",
                    "빠른 프로토타이핑과 반응형 디자인",
                    "풍부한 웹 라이브러리 활용 가능",
                    "크로스 플랫폼 지원 우수"
                ],
                "disadvantages": [
                    "높은 메모리 사용량 (Chrome 기반)",
                    "상대적으로 느린 시작 속도",
                    "네이티브 앱 대비 큰 설치 파일 크기",
                    "Python 데이터 처리 라이브러리 연동 복잡"
                ],
                "tech_details": "Electron + Node.js + React/Vue + SQLite",
                "recommended": False
            },
            {
                "name": ".NET MAUI",
                "subtitle": "마이크로소프트 크로스 플랫폼 framework",
                "description": "C#을 사용하여 Windows, macOS, iOS, Android를 모두 지원하는 애플리케이션을 개발합니다.",
                "advantages": [
                    "뛰어난 성능과 네이티브 사용자 경험",
                    "Microsoft 생태계와의 완벽한 통합",
                    "강력한 타입 시스템과 개발 도구",
                    "엔터프라이즈 환경에서 안정적"
                ],
                "disadvantages": [
                    "상대적으로 높은 학습 곡선",
                    "Python 데이터 처리 라이브러리 대체 필요",
                    "라이센스 비용 (Visual Studio)",
                    "macOS/Linux 개발 환경 제한"
                ],
                "tech_details": "C# + .NET MAUI + SQLite + Office Interop",
                "recommended": False
            }
        ]
    
    def create_stack_option(self, stack, index):
        """기술 스택 옵션 위젯 생성"""
        group_box = QGroupBox()
        layout = QVBoxLayout()
        
        # 제목과 라디오 버튼
        header_layout = QHBoxLayout()
        radio_button = QRadioButton()
        self.button_group.addButton(radio_button, index)
        radio_button.toggled.connect(lambda checked, idx=index: self.on_stack_selected(checked, idx))
        
        title_label = QLabel(stack["name"])
        title_label.setFont(QFont("Arial", 14, QFont.Bold))
        if stack.get("recommended", False):
            title_label.setStyleSheet("color: #2e8b57;")
        
        subtitle_label = QLabel(stack["subtitle"])
        subtitle_label.setFont(QFont("Arial", 10))
        subtitle_label.setStyleSheet("color: #666666;")
        
        header_layout.addWidget(radio_button)
        header_layout.addWidget(title_label)
        header_layout.addWidget(subtitle_label)
        header_layout.addStretch()
        
        layout.addLayout(header_layout)
        
        # 설명
        desc_label = QLabel(stack["description"])
        desc_label.setWordWrap(True)
        desc_label.setStyleSheet("color: #444444; margin: 10px 0;")
        layout.addWidget(desc_label)
        
        # 상세 정보 레이아웃
        details_layout = QHBoxLayout()
        
        # 장점
        advantages_widget = QWidget()
        advantages_layout = QVBoxLayout(advantages_widget)
        advantages_title = QLabel("✅ 장점")
        advantages_title.setFont(QFont("Arial", 11, QFont.Bold))
        advantages_title.setStyleSheet("color: #2e8b57;")
        advantages_layout.addWidget(advantages_title)
        
        for advantage in stack["advantages"]:
            advantage_label = QLabel(f"• {advantage}")
            advantage_label.setWordWrap(True)
            advantage_label.setStyleSheet("color: #555555; margin-left: 10px;")
            advantages_layout.addWidget(advantage_label)
        
        # 단점
        disadvantages_widget = QWidget()
        disadvantages_layout = QVBoxLayout(disadvantages_widget)
        disadvantages_title = QLabel("⚠️ 단점")
        disadvantages_title.setFont(QFont("Arial", 11, QFont.Bold))
        disadvantages_title.setStyleSheet("color: #d2691e;")
        disadvantages_layout.addWidget(disadvantages_title)
        
        for disadvantage in stack["disadvantages"]:
            disadvantage_label = QLabel(f"• {disadvantage}")
            disadvantage_label.setWordWrap(True)
            disadvantage_label.setStyleSheet("color: #555555; margin-left: 10px;")
            disadvantages_layout.addWidget(disadvantage_label)
        
        details_layout.addWidget(advantages_widget)
        details_layout.addWidget(disadvantages_widget)
        layout.addLayout(details_layout)
        
        # 기술 스택 상세
        tech_label = QLabel(f"🔧 기술 스택: {stack['tech_details']}")
        tech_label.setStyleSheet("color: #555555; font-weight: bold; margin-top: 10px; padding: 5px; background-color: #f0f0f0; border-radius: 3px;")
        layout.addWidget(tech_label)
        
        # 권장 표시
        if stack.get("recommended", False):
            recommended_label = QLabel("🌟 권장 선택")
            recommended_label.setStyleSheet("color: #2e8b57; font-weight: bold; margin-top: 5px;")
            layout.addWidget(recommended_label)
        
        group_box.setLayout(layout)
        return group_box
    
    def on_stack_selected(self, checked, index):
        """기술 스택 선택 시 호출"""
        if checked:
            self.selected_stack = self.tech_stacks[index]
            self.select_button.setEnabled(True)
        else:
            self.selected_stack = None
            self.select_button.setEnabled(False)
    
    def confirm_selection(self):
        """선택 확인 및 저장"""
        if not self.selected_stack:
            QMessageBox.warning(self, "경고", "기술 스택을 선택해주세요.")
            return
        
        # 선택 결과 저장
        self.save_selection_result()
        
        # 확인 메시지
        msg = QMessageBox()
        msg.setWindowTitle("기술 스택 선택 완료")
        msg.setIcon(QMessageBox.Information)
        msg.setText(f"선택된 기술 스택: {self.selected_stack['name']}")
        msg.setDetailedText(self.get_next_steps_text())
        msg.setStandardButtons(QMessageBox.Ok)
        msg.exec()
        
        # 애플리케이션 종료
        self.close()
    
    def save_selection_result(self):
        """선택 결과를 파일에 저장"""
        result = {
            "selected_stack": self.selected_stack['name'],
            "timestamp": datetime.now().isoformat(),
            "tech_details": self.selected_stack['tech_details'],
            "description": self.selected_stack['description']
        }
        
        try:
            with open("tech_stack_selection.json", "w", encoding="utf-8") as f:
                json.dump(result, f, ensure_ascii=False, indent=2)
            print(f"기술 스택 선택 결과가 저장되었습니다: {self.selected_stack['name']}")
        except Exception as e:
            print(f"선택 결과 저장 중 오류 발생: {e}")
    
    def get_next_steps_text(self):
        """다음 단계 안내 텍스트 반환"""
        stack_name = self.selected_stack['name']
        
        if "Python" in stack_name:
            return """
다음 단계:
1. 필요한 Python 패키지 설치: pip install -r requirements.txt
2. 기본 프로젝트 구조 생성
3. 데이터베이스 스키마 설계 (SQLite)
4. CSV 데이터 처리 모듈 개발
5. PySide6 기반 메인 UI 개발
6. 엑셀 보고서 생성 모듈 구현
7. 배포용 실행 파일 생성 (PyInstaller)

개발자 확인이 필요하므로 다음 task는 자동으로 진행되지 않습니다.
"""
        elif "Electron" in stack_name:
            return """
다음 단계:
1. Node.js 및 npm 환경 설정
2. Electron 프로젝트 초기화
3. 프론트엔드 프레임워크 선택 (React/Vue)
4. 백엔드 API 서버 구성
5. 데이터베이스 연동 (SQLite)
6. 웹 기반 UI 개발
7. 데이터 처리 및 엑셀 생성 모듈
8. 데스크탑 앱 패키징

개발자 확인이 필요하므로 다음 task는 자동으로 진행되지 않습니다.
"""
        else:  # .NET MAUI
            return """
다음 단계:
1. .NET 8.0 및 Visual Studio 설정
2. MAUI 프로젝트 템플릿 생성
3. 데이터베이스 모델 설계 (Entity Framework)
4. 비즈니스 로직 레이어 구현
5. XAML 기반 UI 개발
6. 데이터 처리 및 Office Interop 연동
7. 플랫폼별 빌드 및 배포

개발자 확인이 필요하므로 다음 task는 자동으로 진행되지 않습니다.
"""

def main():
    """메인 함수"""
    app = QApplication(sys.argv)
    app.setStyle('Fusion')  # 일관된 스타일
    
    # 애플리케이션 정보 설정
    app.setApplicationName("N1VI 기술 스택 선택기")
    app.setApplicationVersion("1.0.0")
    app.setOrganizationName("N1VI Project")
    
    # 메인 창 생성 및 표시
    selector = TechStackSelector()
    selector.show()
    
    # 애플리케이션 실행
    sys.exit(app.exec())

if __name__ == "__main__":
    main() 