#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
N1VI 메인 윈도우 - 실제 세원 A/S 확인서 생성 시스템

실시간 검색, 다중선택, A/S 확인서 생성 기능을 제공합니다.
"""

import logging
import os
import platform
import subprocess
from datetime import datetime
from pathlib import Path
from typing import List, Optional

from PySide6.QtCore import Qt, QTimer, Signal
from PySide6.QtGui import QFont, QIcon
from PySide6.QtWidgets import (
    QApplication, QButtonGroup, QFrame, QGridLayout, QGroupBox,
    QHBoxLayout, QLabel, QLineEdit, QListWidget, QListWidgetItem,
    QMainWindow, QMessageBox, QPushButton, QRadioButton, QScrollArea, QSizePolicy,
    QSplitter, QTextEdit, QVBoxLayout, QWidget
)

from ..models.equipment import Equipment
from ..services.equipment_service import EquipmentService
from ..services.as_service import AsService


logger = logging.getLogger(__name__)


class MainWindow(QMainWindow):
    """N1VI 메인 윈도우 - A/S 확인서 생성 시스템"""
    
    def __init__(self):
        super().__init__()
        
        # 윈도우 기본 설정
        self.setWindowTitle("SHT 확인서 생성시스템")
        self.setMinimumSize(1400, 600)  # 세로 높이를 600px로 축소
        self.setFixedHeight(600)  # 세로 높이 고정
        self.resize(1600, 600)  # 기본 크기를 고정 높이로 설정
        
        # 서비스 초기화
        self.equipment_service = EquipmentService()
        self.as_service = AsService()
        
        # 선택된 설비 목록 (set으로 변경하여 중복 방지 및 빠른 검색)
        self.selected_equipment = set()
        
        # 검색 디바운스 타이머
        self.search_timer = QTimer()
        self.search_timer.setSingleShot(True)
        self.search_timer.timeout.connect(self._perform_search)
        
        # UI 초기화
        self._init_ui()
        
        # 데이터 로드
        self._load_initial_data()
        
        logger.info("N1VI 메인 윈도우 초기화 완료")
        
    def _init_ui(self):
        """UI 초기화"""
        # 중앙 위젯 설정
        central_widget = QWidget()
        self.setCentralWidget(central_widget)
        
        # 메인 레이아웃 (수직)
        main_layout = QVBoxLayout(central_widget)
        main_layout.setContentsMargins(15, 15, 15, 15)  # 마진 줄임
        main_layout.setSpacing(10)  # 간격 줄임
        
        # 헤더 추가
        self._create_header(main_layout)
        
        # 컨텐츠 영역 (좌우 분할)
        content_splitter = QSplitter(Qt.Horizontal)
        main_layout.addWidget(content_splitter)
        
        # 좌측: 검색 영역
        left_widget = self._create_left_panel()
        content_splitter.addWidget(left_widget)
        
        # 우측: A/S 정보 입력 영역
        right_widget = self._create_right_panel()
        content_splitter.addWidget(right_widget)
        
        # 분할 비율 설정 (왼쪽 45%, 오른쪽 55%)
        content_splitter.setSizes([450, 550])
        
        # 툴바 추가
        self._create_toolbar(main_layout)
        
        # TAB 순서 설정
        self._setup_tab_order()
        
    def _create_header(self, parent_layout):
        """헤더 생성"""
        header_frame = QFrame()
        header_frame.setFixedHeight(60)  # 원래 높이로 복원
        header_frame.setStyleSheet("""
            QFrame {
                background: qlineargradient(x1: 0, y1: 0, x2: 1, y2: 0,
                    stop: 0 #2c3e50, stop: 1 #3498db);
                border-radius: 8px;
                margin: 5px;
            }
            QLabel {
                color: white;
                font-size: 20px;  
                font-weight: bold;
                background: transparent;
            }
        """)
        
        header_layout = QHBoxLayout(header_frame)
        header_layout.setContentsMargins(20, 10, 20, 10)
        
        title_label = QLabel("🔧 SHT 확인서 생성시스템")
        header_layout.addWidget(title_label)
        header_layout.addStretch()
        
        parent_layout.addWidget(header_frame)
        
    def _create_left_panel(self) -> QWidget:
        """좌측 검색 패널 생성"""
        left_widget = QWidget()
        left_layout = QVBoxLayout(left_widget)
        left_layout.setContentsMargins(5, 5, 5, 5)  # 마진 줄임
        left_layout.setSpacing(8)  # 간격 줄임
        
        # 검색 그룹
        search_group = QGroupBox("🔍 실시간 검색")
        search_group.setMaximumHeight(100)  # 원래 높이로 복원
        search_layout = QVBoxLayout(search_group)
        search_layout.setContentsMargins(10, 8, 10, 8)  # 원래 마진으로 복원
        search_layout.setSpacing(5)
        
        # 첫 번째 검색창 (기본)
        search1_layout = QHBoxLayout()
        self.txt_search1 = QLineEdit()
        self.txt_search1.setPlaceholderText("기본 검색어 (업체명, 설비번호, 모델명 등)")
        self.txt_search1.setFixedHeight(30)
        self.txt_search1.setStyleSheet("""
            QLineEdit {
                border: 2px solid #bdc3c7;
                border-radius: 8px;
                padding: 5px 10px;
                font-size: 13px;
                background: white;
            }
            QLineEdit:focus {
                border-color: #3498db;
                background: #f8f9fa;
            }
        """)
        self.txt_search1.textChanged.connect(self._on_search_text_changed)
        search1_layout.addWidget(self.txt_search1)
        search_layout.addLayout(search1_layout)
        
        # AND 연산자 및 두 번째 검색창
        search2_layout = QHBoxLayout()
        
        # "그리고" 라벨
        and_label = QLabel("그리고")
        and_label.setFixedWidth(50)
        and_label.setAlignment(Qt.AlignCenter)
        and_label.setStyleSheet("""
            QLabel {
                color: #7f8c8d;
                font-weight: bold;
                font-size: 12px;
                background: #ecf0f1;
                border-radius: 4px;
                padding: 2px;
            }
        """)
        search2_layout.addWidget(and_label)
        
        # 두 번째 검색창 (추가 조건)
        self.txt_search2 = QLineEdit()
        self.txt_search2.setPlaceholderText("추가 검색어 (선택사항 - AND 조건)")
        self.txt_search2.setFixedHeight(30)
        self.txt_search2.setStyleSheet("""
            QLineEdit {
                border: 2px solid #bdc3c7;
                border-radius: 8px;
                padding: 5px 10px;
                font-size: 13px;
                background: white;
            }
            QLineEdit:focus {
                border-color: #e74c3c;
                background: #fdf2f2;
            }
        """)
        self.txt_search2.textChanged.connect(self._on_search_text_changed)
        search2_layout.addWidget(self.txt_search2)
        
        search_layout.addLayout(search2_layout)
        
        left_layout.addWidget(search_group)
        
        # 검색 결과 그룹
        results_group = QGroupBox("📋 검색 결과 (행 클릭으로 선택)")
        # 검색결과 그룹박스 최대 높이를 리스트에 맞게 설정
        results_group.setMaximumHeight(320)  # 리스트 280px + 제목바 및 마진 여유분
        results_layout = QVBoxLayout(results_group)
        results_layout.setContentsMargins(10, 8, 10, 8)  # 원래 마진으로 복원
        results_layout.setSpacing(5)
        
        self.list_results = QListWidget()
        
        # 검색결과 리스트 최대 높이 설정 (세로 길이 70% 축소)
        self.list_results.setMaximumHeight(280)  # 약 70% 크기로 축소
        
        # 다중 선택 모드 설정 (내장 선택 기능과 함께 사용)
        self.list_results.setSelectionMode(QListWidget.MultiSelection)
        
        self.list_results.setStyleSheet("""
            QListWidget {
                border: 1px solid #bdc3c7;
                border-radius: 8px;
                background: white;
                font-size: 12px;
                selection-background-color: #3498db;
                selection-color: white;
            }
            QListWidget::item {
                padding: 8px;
                border-bottom: 1px solid #ecf0f1;
                border-radius: 4px;
                margin: 1px;
            }
            QListWidget::item:hover {
                background: #f1f2f6;
                border: 1px solid #bdc3c7;
            }
            QListWidget::item:selected {
                background: #3498db !important;
                color: white !important;
                border: 2px solid #2980b9 !important;
                font-weight: bold;
            }
        """)
        self.list_results.itemClicked.connect(self._on_item_clicked)
        results_layout.addWidget(self.list_results)
        
        left_layout.addWidget(results_group)
        
        # 선택 현황 표시 (검색결과 그룹 바깥으로 이동)
        self.lbl_selection_status = QLabel("선택됨: 0개")
        self.lbl_selection_status.setStyleSheet("""
            QLabel {
                color: #e74c3c;
                font-weight: bold;
                font-size: 16px;
                padding: 8px;
                background: #fef9e7;
                border: 2px solid #f39c12;
                border-radius: 5px;
            }
        """)
        left_layout.addWidget(self.lbl_selection_status)
        
        return left_widget
        
    def _create_right_panel(self) -> QWidget:
        """우측 A/S 정보 입력 패널 생성"""
        right_widget = QWidget()
        right_layout = QVBoxLayout(right_widget)
        right_layout.setContentsMargins(5, 5, 5, 5)  # 마진 줄임
        right_layout.setSpacing(8)  # 간격 줄임
        
        # A/S 정보 입력 그룹
        as_group = QGroupBox("📋 A/S 확인서 정보 입력")
        as_layout = QVBoxLayout(as_group)
        as_layout.setContentsMargins(10, 8, 10, 8)  # 원래 마진으로 복원
        as_layout.setSpacing(8)  # 원래 간격으로 복원
        
        # 필수 정보 섹션
        required_group = QGroupBox("✅ 필수 정보")
        required_layout = QGridLayout(required_group)
        required_layout.setContentsMargins(8, 5, 8, 5)  # 마진 줄임
        required_layout.setSpacing(6)  # 간격 줄임
        
        # 접수일
        required_layout.addWidget(QLabel("접수일:"), 0, 0)
        self.txt_receipt_date = QLineEdit()
        self.txt_receipt_date.setText(datetime.now().strftime("%Y.%m.%d"))
        self.txt_receipt_date.setFixedHeight(28)  # 높이 줄임
        self.txt_receipt_date.setStyleSheet(self._get_input_style())
        required_layout.addWidget(self.txt_receipt_date, 0, 1)
        
        # 확인자
        required_layout.addWidget(QLabel("확인자:"), 1, 0)
        self.txt_confirmer = QLineEdit()
        self.txt_confirmer.setPlaceholderText("업무수행자 이름을 입력하세요")
        self.txt_confirmer.setFixedHeight(28)
        self.txt_confirmer.setStyleSheet(self._get_input_style())
        required_layout.addWidget(self.txt_confirmer, 1, 1)
        
        # 담당자
        required_layout.addWidget(QLabel("담당자:"), 2, 0)
        self.txt_manager = QLineEdit()
        self.txt_manager.setPlaceholderText("업체의 담당자 이름을 입력하세요")
        self.txt_manager.setFixedHeight(28)
        self.txt_manager.setStyleSheet(self._get_input_style())
        required_layout.addWidget(self.txt_manager, 2, 1)
        
        # 설치구분 (버튼으로 변경)
        required_layout.addWidget(QLabel("설치구분:"), 3, 0)
        install_frame = QFrame()
        install_layout = QHBoxLayout(install_frame)
        install_layout.setContentsMargins(0, 0, 0, 0)
        install_layout.setSpacing(5)
        
        # 선택된 버튼을 추적하기 위한 변수
        self.selected_install_button = None
        
        self.btn_initial = QPushButton("초기(구입)")
        self.btn_repair = QPushButton("고장수리완료") 
        self.btn_temp = QPushButton("임시대용")
        
        # 버튼 스타일링
        button_style = """
            QPushButton {
                font-size: 10px;
                padding: 3px 8px;
                border: 2px solid #bdc3c7;
                border-radius: 4px;
                background: white;
                min-width: 60px;
                max-height: 20px;
            }
            QPushButton:hover {
                border-color: #3498db;
                background: #f8f9fa;
            }
        """
        
        selected_style = """
            QPushButton {
                font-size: 10px;
                padding: 3px 8px;
                border: 2px solid #27ae60;
                border-radius: 4px;
                background: #2ecc71;
                color: white;
                font-weight: bold;
                min-width: 60px;
                max-height: 20px;
            }
        """
        
        for btn in [self.btn_initial, self.btn_repair, self.btn_temp]:
            btn.setStyleSheet(button_style)
            btn.clicked.connect(self._on_install_button_clicked)
            
        install_layout.addWidget(self.btn_initial)
        install_layout.addWidget(self.btn_repair)
        install_layout.addWidget(self.btn_temp)
        install_layout.addStretch()
        
        required_layout.addWidget(install_frame, 3, 1)
        
        as_layout.addWidget(required_group)
        
        # 선택사항 섹션
        optional_group = QGroupBox("⚙️ 선택사항 (빈 값 허용)")
        optional_layout = QGridLayout(optional_group)
        optional_layout.setContentsMargins(8, 5, 8, 5)  # 마진 줄임
        optional_layout.setSpacing(6)  # 간격 줄임
        
        # A/S 신청내역
        optional_layout.addWidget(QLabel("A/S 신청내역:"), 0, 0)
        self.txt_as_request = QTextEdit()
        self.txt_as_request.setPlaceholderText("A/S 신청 내용을 입력하세요 (선택사항)")
        self.txt_as_request.setMaximumHeight(60)  # 높이 줄임
        self.txt_as_request.setStyleSheet(self._get_textarea_style())
        optional_layout.addWidget(self.txt_as_request, 0, 1)
        
        # A/S 수리내역
        optional_layout.addWidget(QLabel("A/S 수리내역:"), 1, 0)
        self.txt_as_repair = QTextEdit()
        self.txt_as_repair.setPlaceholderText("수리 내용을 입력하세요 (선택사항)")
        self.txt_as_repair.setMaximumHeight(60)  # 높이 줄임
        self.txt_as_repair.setStyleSheet(self._get_textarea_style())
        optional_layout.addWidget(self.txt_as_repair, 1, 1)
        
        # 체크사항
        optional_layout.addWidget(QLabel("체크사항:"), 2, 0)
        self.txt_check_items = QTextEdit()
        self.txt_check_items.setPlaceholderText("체크 사항을 입력하세요 (선택사항)")
        self.txt_check_items.setMaximumHeight(60)  # 높이 줄임
        self.txt_check_items.setStyleSheet(self._get_textarea_style())
        optional_layout.addWidget(self.txt_check_items, 2, 1)
        
        as_layout.addWidget(optional_group)
        right_layout.addWidget(as_group)
        
        # 하단 여백 추가
        right_layout.addStretch()
        
        return right_widget
        
    def _create_toolbar(self, parent_layout):
        """툴바 생성"""
        toolbar_frame = QFrame()
        toolbar_frame.setFixedHeight(60)  # 원래 높이로 복원
        toolbar_frame.setStyleSheet("""
            QFrame {
                background: #f8f9fa;
                border: 1px solid #dee2e6;
                border-radius: 8px;
                margin: 5px;
            }
        """)
        
        toolbar_layout = QHBoxLayout(toolbar_frame)
        toolbar_layout.setContentsMargins(15, 8, 15, 8)  # 원래 마진으로 복원
        toolbar_layout.setSpacing(10)
        
        # A/S 확인서 생성 버튼 (왼쪽으로 이동)
        self.btn_generate = QPushButton("📋 A/S 확인서 생성")
        self.btn_generate.setFixedHeight(40)  # 원래 높이로 복원
        self.btn_generate.setStyleSheet("""
            QPushButton {
                background: #28a745;
                color: white;
                border: none;
                border-radius: 8px;
                font-size: 13px;
                font-weight: bold;
                padding: 8px 16px;
            }
            QPushButton:hover {
                background: #218838;
            }
            QPushButton:pressed {
                background: #1e7e34;
            }
            QPushButton:disabled {
                background: #6c757d;
            }
        """)
        self.btn_generate.clicked.connect(self._generate_as_reports)
        toolbar_layout.addWidget(self.btn_generate)
        
        # 출력폴더 열기 버튼 (A/S 확인서 생성 오른쪽으로)
        self.btn_open_folder = QPushButton("📁 출력폴더 열기")
        self.btn_open_folder.setFixedHeight(40)
        self.btn_open_folder.setStyleSheet("""
            QPushButton {
                background: #007bff;
                color: white;
                border: none;
                border-radius: 8px;
                font-size: 13px;
                font-weight: bold;
                padding: 8px 16px;
            }
            QPushButton:hover {
                background: #0056b3;
            }
            QPushButton:pressed {
                background: #004085;
            }
        """)
        self.btn_open_folder.clicked.connect(self._open_output_folder)
        toolbar_layout.addWidget(self.btn_open_folder)
        
        # 구분선
        separator = QFrame()
        separator.setFrameShape(QFrame.VLine)
        separator.setFrameShadow(QFrame.Sunken)
        separator.setStyleSheet("color: #dee2e6;")
        toolbar_layout.addWidget(separator)
        
        # 데이터 재로드 버튼 (출력폴더 열기 오른쪽으로)
        self.btn_reload = QPushButton("🔄 데이터 재로드")
        self.btn_reload.setFixedHeight(40)
        self.btn_reload.setStyleSheet("""
            QPushButton {
                background: #6c757d;
                color: white;
                border: none;
                border-radius: 8px;
                font-size: 13px;
                font-weight: bold;
                padding: 8px 16px;
            }
            QPushButton:hover {
                background: #5a6268;
            }
            QPushButton:pressed {
                background: #495057;
            }
        """)
        self.btn_reload.clicked.connect(self._reload_data)
        toolbar_layout.addWidget(self.btn_reload)
        
        # 구분선
        separator2 = QFrame()
        separator2.setFrameShape(QFrame.VLine)
        separator2.setFrameShadow(QFrame.Sunken)
        separator2.setStyleSheet("color: #dee2e6;")
        toolbar_layout.addWidget(separator2)
        
        # 상태 표시 (오른쪽으로 이동)
        self.lbl_status = QLabel("데이터 로드 중...")
        self.lbl_status.setStyleSheet("""
            QLabel {
                color: #6c757d;
                font-size: 12px;
                font-weight: bold;
            }
        """)
        toolbar_layout.addWidget(self.lbl_status)
        toolbar_layout.addStretch()
        
        parent_layout.addWidget(toolbar_frame)
        
    def _get_input_style(self) -> str:
        """입력 필드 스타일"""
        return """
            QLineEdit {
                border: 1px solid #ced4da;
                border-radius: 6px;
                padding: 4px 8px;
                font-size: 12px;
                background: white;
            }
            QLineEdit:focus {
                border-color: #80bdff;
                box-shadow: 0 0 0 0.2rem rgba(0,123,255,.25);
            }
        """
        
    def _get_textarea_style(self) -> str:
        """텍스트 영역 스타일"""
        return """
            QTextEdit {
                border: 1px solid #ced4da;
                border-radius: 6px;
                padding: 4px 8px;
                font-size: 12px;
                background: white;
            }
            QTextEdit:focus {
                border-color: #80bdff;
            }
        """
    
    def _setup_tab_order(self):
        """TAB 순서 설정"""
        widgets = [
            self.txt_search1,          # 기본 검색어 입력창
            self.txt_search2,          # 추가 검색어 입력창
            self.txt_receipt_date,     # 접수일 입력창
            self.txt_confirmer,        # 확인자 입력창
            self.txt_manager,          # 담당자 입력창
            self.txt_as_request,       # A/S 신청내역 입력창
            self.txt_as_repair,        # A/S 수리내역 입력창
            self.txt_check_items,      # 체크사항 입력창
            self.btn_generate,         # A/S 확인서 생성 버튼
            self.btn_open_folder,      # 출력폴더 열기
        ]
        
        for i in range(len(widgets) - 1):
            self.setTabOrder(widgets[i], widgets[i + 1])
        
        # 마지막에서 처음으로 순환
        self.setTabOrder(widgets[-1], widgets[0])
        
    def _load_initial_data(self):
        """초기 데이터 로드"""
        if self.equipment_service.load_data():
            equipment_count = len(self.equipment_service.get_all_equipment())
            self.lbl_status.setText(f"✅ 준비 완료 - {equipment_count}개 설비 로드됨")
            logger.info(f"초기 데이터 로드 성공: {equipment_count}개 설비")
        else:
            self.lbl_status.setText("❌ 데이터 로드 실패")
            QMessageBox.warning(
                self,
                "데이터 로드 오류",
                "설비 데이터를 로드할 수 없습니다.\n데이터 파일을 확인해주세요."
            )
    
    def _on_search_text_changed(self):
        """검색어 변경 시 디바운스 타이머 시작"""
        self.search_timer.stop()
        self.search_timer.start(300)  # 300ms 대기
        
    def _perform_search(self):
        """실제 검색 수행 - AND 연산 지원"""
        query1 = self.txt_search1.text().strip()
        query2 = self.txt_search2.text().strip()
        
        # 검색 결과 초기화
        self.list_results.clear()
        
        if not query1:
            self.lbl_status.setText("기본 검색어를 입력하세요")
            return
        
        # 기본 검색 수행
        results = self.equipment_service.search_equipment(query1)
        
        # 추가 검색어가 있으면 AND 연산 수행
        if query2:
            # 기본 검색 결과에서 추가 조건을 만족하는 항목만 필터링
            filtered_results = []
            for equipment in results:
                # 모든 필드를 대상으로 추가 검색어 확인
                searchable_text = " ".join(filter(None, [
                    equipment.company_name,
                    equipment.machine_number,
                    equipment.serial_number,
                    equipment.model_name,
                    equipment.manager,
                    equipment.manufacturing_date
                ])).lower()
                
                if query2.lower() in searchable_text:
                    filtered_results.append(equipment)
            
            results = filtered_results
        
        # 결과 표시
        logger.info(f"검색 결과 표시 시작 - 기존 선택된 설비: {len(self.selected_equipment)}개")
        
        # 업데이트 일시 중단
        self.list_results.setUpdatesEnabled(False)
        
        for equipment in results:
            # 선택 여부 먼저 확인
            is_selected = False
            for selected_eq in self.selected_equipment:
                if (equipment.company_name == selected_eq.company_name and 
                    equipment.machine_number == selected_eq.machine_number and
                    equipment.serial_number == selected_eq.serial_number):
                    is_selected = True
                    break
            
            # 표시 텍스트 구성
            display_text = f"🏢 {equipment.company_name} | 🔧 {equipment.machine_number}"
            if equipment.model_name:
                display_text += f" | 📋 {equipment.model_name}"
            if equipment.serial_number:
                display_text += f" | 🏷️ {equipment.serial_number}"
            if equipment.manager:
                display_text += f" | 👤 {equipment.manager}"
            
            # 리스트 아이템 생성
            item = QListWidgetItem(display_text)
            item.setData(Qt.UserRole, equipment)  # 설비 객체 저장
            
            # 선택 상태 표시 (다중 방법 적용)
            if is_selected:
                item.setBackground(Qt.cyan)
                item.setData(Qt.BackgroundRole, Qt.cyan)
                item.setSelected(True)  # QListWidget 내장 선택 설정
                logger.info(f"선택 상태 복원: {equipment.company_name} - {equipment.machine_number}")
            else:
                item.setBackground(Qt.white)
                item.setData(Qt.BackgroundRole, Qt.white)
                item.setSelected(False)  # QListWidget 내장 선택 해제
                logger.debug(f"선택되지 않음: {equipment.company_name} - {equipment.machine_number}")
                
            self.list_results.addItem(item)
        
        # 업데이트 재개 및 강제 리페인트
        self.list_results.setUpdatesEnabled(True)
        self.list_results.repaint()
        QApplication.processEvents()
        
        # 검색 결과 표시 완료 후 선택 상태 강제 재적용
        self._refresh_selection_display()
        
        # 상태 업데이트
        if query2:
            search_info = f"'{query1}' AND '{query2}'"
        else:
            search_info = f"'{query1}'"
        
        logger.info(f"검색 완료: {search_info} -> {len(results)}건 발견")
        self.lbl_status.setText(f"검색 완료: {len(results)}건 발견")
        
    def _on_install_button_clicked(self):
        """설치구분 버튼 클릭 처리"""
        sender = self.sender()
        
        # 모든 버튼을 기본 스타일로 리셋
        button_style = """
            QPushButton {
                font-size: 10px;
                padding: 3px 8px;
                border: 2px solid #bdc3c7;
                border-radius: 4px;
                background: white;
                min-width: 60px;
                max-height: 20px;
            }
            QPushButton:hover {
                border-color: #3498db;
                background: #f8f9fa;
            }
        """
        
        selected_style = """
            QPushButton {
                font-size: 10px;
                padding: 3px 8px;
                border: 2px solid #27ae60;
                border-radius: 4px;
                background: #2ecc71;
                color: white;
                font-weight: bold;
                min-width: 60px;
                max-height: 20px;
            }
        """
        
        # 모든 버튼을 기본 스타일로
        for btn in [self.btn_initial, self.btn_repair, self.btn_temp]:
            btn.setStyleSheet(button_style)
        
        # 클릭된 버튼을 선택 스타일로
        sender.setStyleSheet(selected_style)
        self.selected_install_button = sender
    
    def _on_item_clicked(self, item: QListWidgetItem):
        """검색 결과 아이템 클릭 시 선택/해제"""
        equipment = item.data(Qt.UserRole)
        
        # 디버깅 로그
        logger.info(f"행 클릭: {equipment.company_name} - {equipment.machine_number} (ID: {id(equipment)})")
        logger.info(f"현재 선택된 설비 수: {len(self.selected_equipment)}")
        
        # 더 확실한 비교를 위해 직접 찾기
        found_equipment = None
        for selected_eq in self.selected_equipment:
            if (equipment.company_name == selected_eq.company_name and 
                equipment.machine_number == selected_eq.machine_number and
                equipment.serial_number == selected_eq.serial_number):
                found_equipment = selected_eq
                break
        
        if found_equipment:
            # 선택 해제
            self.selected_equipment.remove(found_equipment)
            item.setSelected(False)  # QListWidget 내장 선택 해제
            logger.info(f"선택 해제: {equipment.company_name} - {equipment.machine_number}")
        else:
            # 선택 추가
            self.selected_equipment.add(equipment)
            item.setSelected(True)   # QListWidget 내장 선택 설정
            logger.info(f"선택 추가: {equipment.company_name} - {equipment.machine_number}")
        
        logger.info(f"선택 후 설비 수: {len(self.selected_equipment)}")
        
        # 전체 선택 상태 재적용 (모든 아이템의 색상을 올바르게 표시)
        self._refresh_selection_display()
        
    def _refresh_selection_display(self):
        """현재 리스트의 모든 아이템에 대해 선택 상태를 강제로 재적용"""
        logger.info(f"=== 선택 상태 재적용 시작 ===")
        logger.info(f"현재 선택된 설비 수: {len(self.selected_equipment)}")
        
        # 선택된 설비 목록 출력
        for selected_eq in self.selected_equipment:
            logger.info(f"선택된 설비: {selected_eq.company_name} - {selected_eq.machine_number} (ID: {id(selected_eq)})")
        
        # QListWidget 업데이트 일시 중단 (성능 및 깜빡임 방지)
        self.list_results.setUpdatesEnabled(False)
        
        for i in range(self.list_results.count()):
            item = self.list_results.item(i)
            equipment = item.data(Qt.UserRole)
            
            logger.info(f"리스트 아이템 {i}: {equipment.company_name} - {equipment.machine_number} (ID: {id(equipment)})")
            
            # 더 정확한 비교를 위해 직접 매칭 확인
            is_selected = False
            for selected_eq in self.selected_equipment:
                if (equipment.company_name == selected_eq.company_name and 
                    equipment.machine_number == selected_eq.machine_number and
                    equipment.serial_number == selected_eq.serial_number):
                    is_selected = True
                    logger.info(f"  → 매칭됨! 파란색으로 표시")
                    break
            
            # Qt의 다양한 방식으로 배경색 설정 (더 확실한 방법)
            if is_selected:
                # 방법 1: setBackground
                item.setBackground(Qt.cyan)
                # 방법 2: setData with BackgroundRole  
                item.setData(Qt.BackgroundRole, Qt.cyan)
                # 방법 3: QListWidget 내장 선택 설정
                item.setSelected(True)
            else:
                item.setBackground(Qt.white)
                item.setData(Qt.BackgroundRole, Qt.white)
                item.setSelected(False)
                logger.info(f"  → 매칭되지 않음, 흰색으로 표시")
        
        # QListWidget 업데이트 재개 및 강제 리페인트
        self.list_results.setUpdatesEnabled(True)
        self.list_results.repaint()
        
        # Qt 이벤트 루프 강제 처리
        QApplication.processEvents()
        
        # 선택 현황도 업데이트
        self._update_selection_status()
        logger.info(f"=== 선택 상태 재적용 완료 ===")
        logger.info(f"총 {len(self.selected_equipment)}개 선택됨")
    
    def _update_selection_status(self):
        """선택 현황 표시 업데이트 (선택된 설비 상세 정보 포함)"""
        count = len(self.selected_equipment)
        
        # 기본 선택 개수 표시
        status_text = f"선택됨: {count}개"
        
        # 선택된 설비가 있으면 상세 정보도 표시
        if count > 0:
            selected_names = []
            for equipment in list(self.selected_equipment)[:3]:  # 최대 3개만 표시
                selected_names.append(f"{equipment.company_name}-{equipment.machine_number}")
            
            if count > 3:
                status_text += f" ({', '.join(selected_names)}, 외 {count-3}개)"
            else:
                status_text += f" ({', '.join(selected_names)})"
        
        self.lbl_selection_status.setText(status_text)
        
        # A/S 생성 버튼 활성화/비활성화
        self.btn_generate.setEnabled(count > 0)
        
    def _generate_as_reports(self):
        """A/S 확인서 생성"""
        if not self.selected_equipment:
            QMessageBox.warning(self, "경고", "설비를 선택해주세요.")
            return
        
        # 필수 입력 확인
        if not self.txt_receipt_date.text().strip():
            QMessageBox.warning(self, "경고", "접수일을 입력해주세요.")
            self.txt_receipt_date.setFocus()
            return
            
        if not self.txt_confirmer.text().strip():
            QMessageBox.warning(self, "경고", "확인자를 입력해주세요.")
            self.txt_confirmer.setFocus()
            return
        
        # 설치구분 선택 확인
        if not self.selected_install_button:
            QMessageBox.warning(self, "경고", "설치구분을 선택해주세요.")
            return
        
        # A/S 입력 정보 수집
        as_inputs = {
            'receipt_date': self.txt_receipt_date.text().strip(),
            'as_request': self.txt_as_request.toPlainText().strip(),
            'as_repair': self.txt_as_repair.toPlainText().strip(),
            'check_items': self.txt_check_items.toPlainText().strip(),
            'confirmer': self.txt_confirmer.text().strip(),
            'manager': self.txt_manager.text().strip(),
            'install_type': self._get_selected_install_type()
        }
        
        try:
            # 진행 상황 표시
            self.lbl_status.setText(f"A/S 확인서 생성 중... (0/{len(self.selected_equipment)})")
            QApplication.processEvents()
            
            # 다중 A/S 확인서 생성
            success_count, success_files, error_messages = self.as_service.generate_multiple_reports(
                list(self.selected_equipment), 
                as_inputs
            )
            
            # 결과 표시
            if success_count == len(self.selected_equipment):
                QMessageBox.information(
                    self,
                    "생성 완료",
                    f"✅ {success_count}개의 A/S 확인서가 성공적으로 생성되었습니다!"
                )
                self.lbl_status.setText(f"✅ {success_count}개 A/S 확인서 생성 완료")
            else:
                # 부분 성공 또는 완전 실패
                failed_count = len(self.selected_equipment) - success_count
                error_detail = "\n".join(error_messages[:3])  # 최대 3개 오류만 표시
                
                QMessageBox.warning(
                    self,
                    "생성 완료 (일부 실패)",
                    f"✅ 성공: {success_count}개\n"
                    f"❌ 실패: {failed_count}개\n\n"
                    f"실패 원인:\n{error_detail}"
                )
                self.lbl_status.setText(f"⚠️ {success_count}개 성공, {failed_count}개 실패")
            
        except Exception as e:
            logger.error(f"A/S 확인서 생성 중 예외 발생: {e}", exc_info=True)
            QMessageBox.critical(
                self,
                "오류",
                f"A/S 확인서 생성 중 오류가 발생했습니다:\n{e}"
            )
            self.lbl_status.setText("❌ A/S 확인서 생성 실패")
    
    def _get_selected_install_type(self) -> str:
        """선택된 설치구분 반환"""
        if self.selected_install_button == self.btn_initial:
            return "초기"
        elif self.selected_install_button == self.btn_repair:
            return "고장수리완료"
        elif self.selected_install_button == self.btn_temp:
            return "임시대용"
        return ""
    
    def _open_output_folder(self):
        """출력 폴더 열기"""
        output_dir = Path("output")
        
        if not output_dir.exists():
            output_dir.mkdir(parents=True, exist_ok=True)
        
        try:
            if platform.system() == "Windows":
                os.startfile(str(output_dir))
            elif platform.system() == "Darwin":  # macOS
                subprocess.run(["open", str(output_dir)])
            else:  # Linux
                subprocess.run(["xdg-open", str(output_dir)])
        except Exception as e:
            QMessageBox.warning(
                self,
                "폴더 열기 실패",
                f"출력 폴더를 열 수 없습니다:\n{e}"
            )
    
    def _reload_data(self):
        """데이터 재로드"""
        self.lbl_status.setText("데이터 재로드 중...")
        
        # 선택 초기화
        self.selected_equipment.clear()
        self.list_results.clear()
        self.txt_search1.clear()
        self.txt_search2.clear()
        self._update_selection_status()
        
        # 데이터 재로드
        if self.equipment_service.reload_data():
            equipment_count = len(self.equipment_service.get_all_equipment())
            self.lbl_status.setText(f"✅ 재로드 완료 - {equipment_count}개 설비")
            QMessageBox.information(
                self,
                "재로드 완료",
                f"데이터가 성공적으로 재로드되었습니다.\n설비 수: {equipment_count}개"
            )
        else:
            self.lbl_status.setText("❌ 데이터 재로드 실패")
            QMessageBox.warning(
                self,
                "재로드 실패",
                "데이터 재로드에 실패했습니다.\n파일을 확인해주세요."
            ) 