#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
설비 정보 검색 서비스

CSV 파일에서 설비 정보를 로드하고 검색하는 기능을 제공합니다.
"""

import pandas as pd
import time
import os
from pathlib import Path
from typing import List, Optional, Union
import logging

from ..models.equipment import Equipment, EquipmentSearchResult

# 로깅 설정
logger = logging.getLogger(__name__)

class EquipmentService:
    """설비 정보 검색 서비스
    
    CSV 파일에서 설비 정보를 로드하고 빠른 검색을 제공합니다.
    PRD 요구사항: 평균 조회 시간 1초 이내
    """
    
    def __init__(self, csv_file_path: Optional[str] = None):
        """서비스 초기화
        
        Args:
            csv_file_path: CSV 파일 경로 (없으면 기본 경로 사용)
        """
        self.csv_file_path = csv_file_path or self._get_default_csv_path()
        self.equipment_list: List[Equipment] = []
        self.last_loaded_time: Optional[float] = None
        self.load_error: Optional[str] = None
        
        # CSV 파일이 존재하면 자동 로드
        if os.path.exists(self.csv_file_path):
            self.load_data()
    
    def _get_default_csv_path(self) -> str:
        """기본 CSV 파일 경로 반환"""
        # 프로젝트 루트의 data 폴더에서 찾기
        project_root = Path(__file__).parent.parent.parent.parent
        default_path = project_root / "data" / "equipment_data.csv"
        return str(default_path)
    
    def load_data(self, csv_file_path: Optional[str] = None) -> bool:
        """CSV 파일에서 설비 데이터 로드
        
        Args:
            csv_file_path: CSV 파일 경로 (없으면 기존 경로 사용)
        
        Returns:
            bool: 로드 성공 여부
        """
        if csv_file_path:
            self.csv_file_path = csv_file_path
        
        try:
            start_time = time.time()
            
            # 파일 존재 확인
            if not os.path.exists(self.csv_file_path):
                self.load_error = f"CSV 파일을 찾을 수 없습니다: {self.csv_file_path}"
                logger.error(self.load_error)
                return False
            
            # CSV 파일 읽기 (다양한 인코딩 시도)
            encodings = ['utf-8', 'cp949', 'euc-kr', 'utf-8-sig']
            df = None
            
            for encoding in encodings:
                try:
                    df = pd.read_csv(self.csv_file_path, encoding=encoding)
                    logger.info(f"CSV 파일 로드 성공 (인코딩: {encoding})")
                    break
                except UnicodeDecodeError:
                    continue
                except Exception as e:
                    logger.warning(f"인코딩 {encoding}으로 로드 실패: {e}")
                    continue
            
            if df is None:
                self.load_error = f"CSV 파일 인코딩을 인식할 수 없습니다: {self.csv_file_path}"
                logger.error(self.load_error)
                return False
            
            # 빈 DataFrame 확인
            if df.empty:
                self.load_error = "CSV 파일이 비어있습니다."
                logger.error(self.load_error)
                return False
            
            # Equipment 객체들로 변환
            equipment_list = []
            failed_rows = 0
            
            for index, row in df.iterrows():
                try:
                    # NaN 값을 None으로 변환
                    row_dict = row.to_dict()
                    for key, value in row_dict.items():
                        if pd.isna(value):
                            row_dict[key] = None
                        elif isinstance(value, str):
                            row_dict[key] = value.strip() if value.strip() else None
                    
                    equipment = Equipment.from_csv_row(row_dict)
                    equipment_list.append(equipment)
                    
                except Exception as e:
                    failed_rows += 1
                    logger.warning(f"행 {index + 1} 처리 실패: {e}")
                    continue
            
            self.equipment_list = equipment_list
            self.last_loaded_time = time.time()
            self.load_error = None
            
            load_time = self.last_loaded_time - start_time
            success_rows = len(equipment_list)
            total_rows = len(df)
            
            logger.info(f"데이터 로드 완료: {success_rows}/{total_rows}행 성공, "
                       f"실패 {failed_rows}행, 소요시간 {load_time:.3f}초")
            
            return True
            
        except Exception as e:
            self.load_error = f"데이터 로드 중 오류 발생: {e}"
            logger.error(self.load_error)
            return False
    
    def search_equipment_detailed(self, query: str, limit: int = 10) -> EquipmentSearchResult:
        """설비 정보 검색
        
        Args:
            query: 검색어 (회사명+설비명 또는 회사명+시리얼넘버)
            limit: 최대 검색 결과 수
        
        Returns:
            EquipmentSearchResult: 검색 결과
        """
        start_time = time.time()
        
        # 데이터가 로드되지 않은 경우
        if not self.equipment_list:
            if self.load_error:
                logger.error(f"검색 실패: {self.load_error}")
            else:
                logger.warning("검색 데이터가 로드되지 않았습니다.")
            
            search_time = time.time() - start_time
            return EquipmentSearchResult(query, [], search_time)
        
        # 검색어 전처리
        if not query or not query.strip():
            search_time = time.time() - start_time
            return EquipmentSearchResult(query, [], search_time)
        
        query = query.strip()
        
        # 검색 수행
        matching_equipment = []
        
        for equipment in self.equipment_list:
            if equipment.matches_search(query):
                matching_equipment.append(equipment)
                
                # 제한된 결과 수에 도달하면 중단
                if len(matching_equipment) >= limit:
                    break
        
        search_time = time.time() - start_time
        
        logger.info(f"검색 완료: '{query}' -> {len(matching_equipment)}건 발견, "
                   f"소요시간 {search_time:.3f}초")
        
        return EquipmentSearchResult(query, matching_equipment, search_time)
    
    def search_by_company(self, company_name: str, limit: int = 50) -> EquipmentSearchResult:
        """회사명으로 모든 설비 검색
        
        Args:
            company_name: 회사명
            limit: 최대 검색 결과 수
        
        Returns:
            EquipmentSearchResult: 검색 결과
        """
        start_time = time.time()
        
        if not self.equipment_list:
            search_time = time.time() - start_time
            return EquipmentSearchResult(company_name, [], search_time)
        
        if not company_name or not company_name.strip():
            search_time = time.time() - start_time
            return EquipmentSearchResult(company_name, [], search_time)
        
        company_name = company_name.strip().lower()
        matching_equipment = []
        
        for equipment in self.equipment_list:
            if equipment.company_name.lower() == company_name:
                matching_equipment.append(equipment)
                
                if len(matching_equipment) >= limit:
                    break
        
        search_time = time.time() - start_time
        
        logger.info(f"회사별 검색 완료: '{company_name}' -> {len(matching_equipment)}건 발견, "
                   f"소요시간 {search_time:.3f}초")
        
        return EquipmentSearchResult(f"회사: {company_name}", matching_equipment, search_time)
    
    def get_all_equipment(self) -> List[Equipment]:
        """모든 설비 목록 반환
        
        Returns:
            List[Equipment]: 로드된 모든 설비 목록
        """
        return self.equipment_list.copy()
    
    def search_equipment(self, query: str) -> List[Equipment]:
        """실시간 검색용 설비 검색 (간단 버전)
        
        Args:
            query: 검색어
        
        Returns:
            List[Equipment]: 검색된 설비 목록
        """
        if not self.equipment_list or not query or not query.strip():
            return []
        
        query = query.strip().lower()
        matching_equipment = []
        
        for equipment in self.equipment_list:
            # 모든 필드에서 부분 일치 검색 (회사명, 설비번호, 시리얼넘버, 모델명, 담당자)
            searchable_text = " ".join(filter(None, [
                equipment.company_name,
                equipment.machine_number,
                equipment.serial_number,
                equipment.model_name,
                equipment.manager,
                equipment.manufacturing_date
            ])).lower()
            
            if query in searchable_text:
                matching_equipment.append(equipment)
        
        logger.info(f"실시간 검색 완료: '{query}' -> {len(matching_equipment)}건 발견")
        return matching_equipment
    
    def get_equipment_by_serial(self, serial_number: str) -> Optional[Equipment]:
        """시리얼넘버로 정확한 설비 정보 조회
        
        Args:
            serial_number: 시리얼넘버
        
        Returns:
            Equipment or None: 찾은 설비 정보
        """
        if not self.equipment_list or not serial_number:
            return None
        
        serial_number = serial_number.strip()
        
        for equipment in self.equipment_list:
            if equipment.serial_number and equipment.serial_number == serial_number:
                return equipment
        
        return None
    
    def get_company_names(self) -> List[str]:
        """모든 회사명 목록 반환 (자동완성용)
        
        Returns:
            List[str]: 중복 제거된 회사명 목록
        """
        if not self.equipment_list:
            return []
        
        company_names = set()
        for equipment in self.equipment_list:
            if equipment.company_name:
                company_names.add(equipment.company_name)
        
        return sorted(list(company_names))
    
    def get_machine_numbers_by_company(self, company_name: str) -> List[str]:
        """특정 회사의 모든 설비번호 목록 반환
        
        Args:
            company_name: 회사명
        
        Returns:
            List[str]: 해당 회사의 설비번호 목록
        """
        if not self.equipment_list or not company_name:
            return []
        
        company_name = company_name.strip()
        machine_numbers = set()
        
        for equipment in self.equipment_list:
            if (equipment.company_name == company_name and 
                equipment.machine_number):
                machine_numbers.add(equipment.machine_number)
        
        return sorted(list(machine_numbers))
    
    def get_data_statistics(self) -> dict:
        """데이터 통계 정보 반환
        
        Returns:
            dict: 데이터 통계 정보
        """
        if not self.equipment_list:
            return {
                'total_count': 0,
                'company_count': 0,
                'loaded_time': None,
                'csv_file': self.csv_file_path,
                'load_error': self.load_error
            }
        
        # 통계 계산
        company_names = set()
        missing_fields_count = 0
        
        for equipment in self.equipment_list:
            company_names.add(equipment.company_name)
            if equipment.has_missing_fields():
                missing_fields_count += 1
        
        return {
            'total_count': len(self.equipment_list),
            'company_count': len(company_names),
            'missing_fields_count': missing_fields_count,
            'loaded_time': self.last_loaded_time,
            'csv_file': self.csv_file_path,
            'load_error': self.load_error
        }
    
    def reload_data(self) -> bool:
        """데이터 다시 로드
        
        Returns:
            bool: 재로드 성공 여부
        """
        logger.info("데이터 재로드 시작...")
        return self.load_data()
    
    def is_data_loaded(self) -> bool:
        """데이터가 로드되었는지 확인
        
        Returns:
            bool: 데이터 로드 상태
        """
        return len(self.equipment_list) > 0 and self.load_error is None 