#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
A/S 확인서 생성 서비스

실제 세원 A/S 확인서 엑셀 파일을 기반으로 새로운 A/S 확인서를 생성합니다.
"""

import logging
import shutil
from pathlib import Path
from typing import List
from datetime import datetime

from openpyxl import load_workbook

from ..models.equipment import Equipment
from ..models.as_report import AsReportData, AsReportSettings


logger = logging.getLogger(__name__)


class AsService:
    """A/S 확인서 생성 서비스"""
    
    def __init__(self):
        """A/S 서비스 초기화"""
        self.project_root = Path(__file__).parent.parent.parent.parent
        self.output_dir = self.project_root / "output"
        self.template_path = self.project_root / "세원_SHT-300VX_1호기_AS 확인서_250708.xlsx"
        self.settings = AsReportSettings()
        
        # 출력 폴더 생성
        self.output_dir.mkdir(exist_ok=True)
        
        logger.info(f"A/S 서비스 초기화 완료 - 템플릿: {self.template_path}")
    
    def validate_template(self) -> bool:
        """템플릿 파일 존재 여부 확인
        
        Returns:
            bool: 템플릿 파일 존재 여부
        """
        if not self.template_path.exists():
            logger.error(f"템플릿 파일을 찾을 수 없습니다: {self.template_path}")
            return False
        
        logger.info(f"템플릿 파일 확인 완료: {self.template_path}")
        return True
    
    def create_equipment_data(self, equipment: Equipment) -> AsReportData:
        """설비 정보를 바탕으로 A/S 데이터 생성
        
        Args:
            equipment (Equipment): 설비 정보
            
        Returns:
            AsReportData: A/S 확인서 데이터
        """
        as_data = AsReportData(
            company_name=equipment.company_name,
            manufacturing_date=equipment.manufacturing_date or "",
            model_name=equipment.model_name,
            machine_number=equipment.machine_number or "",  # 설비번호 추가
            serial_number=equipment.serial_number or "",
            manager=equipment.manager or ""  # 설비 데이터의 담당자 사용 (없으면 빈 값)
        )
        
        logger.info(f"A/S 데이터 생성 완료: {equipment.company_name} - {equipment.machine_number}")
        return as_data
    
    def generate_as_report(self, as_data: AsReportData) -> tuple[bool, str, str]:
        """A/S 확인서 생성
        
        Args:
            as_data (AsReportData): A/S 확인서 데이터
            
        Returns:
            tuple[bool, str, str]: (성공 여부, 파일 경로, 오류 메시지)
        """
        start_time = datetime.now()
        
        try:
            # 필수 필드 검증
            is_valid, errors = as_data.validate_required_fields()
            if not is_valid:
                error_msg = "필수 정보가 누락되었습니다: " + ", ".join(errors)
                logger.error(error_msg)
                return False, "", error_msg
            
            # 템플릿 파일 검증
            if not self.validate_template():
                return False, "", "템플릿 파일을 찾을 수 없습니다"
            
            # 출력 파일 경로 생성
            output_filename = as_data.get_filename()
            output_path = self.output_dir / output_filename
            
            # 템플릿 파일 복사
            logger.info(f"템플릿 파일 복사 시작: {self.template_path} -> {output_path}")
            shutil.copy2(self.template_path, output_path)
            logger.info(f"템플릿 파일 복사 완료: {output_path}")
            
            # 엑셀 파일 열기 및 데이터 입력
            workbook = load_workbook(output_path)
            worksheet = workbook.active
            
            # 셀 매핑 데이터 가져오기
            cell_mapping = as_data.get_cell_mapping()
            
            # 각 셀에 데이터 입력
            updated_cells = []
            for cell_addr, value in cell_mapping.items():
                if value:  # 빈 값이 아닌 경우만 업데이트
                    try:
                        worksheet[cell_addr].value = value
                        updated_cells.append(f"{cell_addr}={value}")
                        logger.debug(f"셀 업데이트: {cell_addr} = {value}")
                    except Exception as e:
                        logger.warning(f"셀 업데이트 실패: {cell_addr} = {value}, 오류: {e}")
                else:
                    # 빈 값인 경우 셀 내용 지우기 (선택사항 필드들)
                    if cell_addr in ['G9', 'G17', 'G25']:  # A/S 신청내역, 수리내역, 체크사항
                        try:
                            worksheet[cell_addr].value = ""
                            updated_cells.append(f"{cell_addr}=<빈값>")
                            logger.debug(f"셀 지우기: {cell_addr}")
                        except Exception as e:
                            logger.warning(f"셀 지우기 실패: {cell_addr}, 오류: {e}")
            
            # 파일 저장
            workbook.save(output_path)
            workbook.close()
            
            # 성공 로그
            elapsed_time = (datetime.now() - start_time).total_seconds()
            logger.info(f"A/S 확인서 생성 완료: {output_path} (소요시간: {elapsed_time:.3f}초)")
            logger.info(f"업데이트된 셀: {len(updated_cells)}개 - {', '.join(updated_cells[:5])}...")
            
            return True, str(output_path), ""
            
        except Exception as e:
            error_msg = f"A/S 확인서 생성 실패: {e}"
            logger.error(error_msg, exc_info=True)
            return False, "", error_msg
    
    def generate_multiple_reports(self, equipment_list: List[Equipment], 
                                as_inputs: dict) -> tuple[int, List[str], List[str]]:
        """다중 설비에 대한 A/S 확인서 일괄 생성
        
        Args:
            equipment_list (List[Equipment]): 설비 목록
            as_inputs (dict): 사용자 입력 A/S 정보
            
        Returns:
            tuple[int, List[str], List[str]]: (성공 개수, 성공 파일 목록, 실패 메시지 목록)
        """
        start_time = datetime.now()
        success_files = []
        error_messages = []
        
        logger.info(f"다중 A/S 확인서 생성 시작: {len(equipment_list)}개 설비")
        
        for equipment in equipment_list:
            try:
                # A/S 데이터 생성
                as_data = self.create_equipment_data(equipment)
                
                # 사용자 입력 데이터 적용
                as_data.receipt_date = as_inputs.get('receipt_date', '')
                as_data.as_request = as_inputs.get('as_request', '')
                as_data.as_repair = as_inputs.get('as_repair', '')
                as_data.check_items = as_inputs.get('check_items', '')
                as_data.confirmer = as_inputs.get('confirmer', '')
                
                # 담당자 정보가 입력된 경우 적용
                if as_inputs.get('manager'):
                    as_data.manager = as_inputs['manager']
                
                # 설치구분 정보 적용
                if as_inputs.get('install_type'):
                    as_data.install_type = as_inputs['install_type']
                
                # A/S 확인서 생성
                success, file_path, error_msg = self.generate_as_report(as_data)
                
                if success:
                    success_files.append(file_path)
                    logger.info(f"성공: {equipment.company_name} - {equipment.machine_number}")
                else:
                    error_msg = f"{equipment.company_name} - {equipment.machine_number}: {error_msg}"
                    error_messages.append(error_msg)
                    logger.error(f"실패: {error_msg}")
                    
            except Exception as e:
                error_msg = f"{equipment.company_name} - {equipment.machine_number}: {e}"
                error_messages.append(error_msg)
                logger.error(f"예외 발생: {error_msg}", exc_info=True)
        
        # 결과 로그
        elapsed_time = (datetime.now() - start_time).total_seconds()
        logger.info(f"다중 A/S 확인서 생성 완료: {len(success_files)}개 성공, {len(error_messages)}개 실패 (소요시간: {elapsed_time:.3f}초)")
        
        return len(success_files), success_files, error_messages
    
    def get_default_manager(self) -> str:
        """기본 담당자 정보 반환
        
        Returns:
            str: 기본 담당자명
        """
        return self.settings.default_manager 