#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
설비 정보 데이터 모델

설비 정보를 표현하고 검증하는 클래스들을 정의합니다.
"""

from dataclasses import dataclass
from datetime import datetime
from typing import Optional, Dict, Any
import re

@dataclass
class Equipment:
    """설비 정보 모델
    
    PRD 요구사항에 따른 설비 정보 필드:
    - 회사명 (company_name)
    - 설비번호 (machine_number) 
    - 시리얼넘버 (serial_number)
    - 제작일 (manufacturing_date)
    - 모델명 (model_name)
    - 담당자 (manager)
    """
    
    company_name: str
    machine_number: Optional[str] = None
    serial_number: Optional[str] = None
    manufacturing_date: Optional[str] = None
    model_name: Optional[str] = None
    manager: Optional[str] = None
    
    def __post_init__(self):
        """초기화 후 데이터 검증 및 정규화"""
        # 회사명은 필수 필드
        if not self.company_name or not self.company_name.strip():
            raise ValueError("회사명은 필수 입력 항목입니다.")
        
        # 공백 제거 및 정규화
        self.company_name = self.company_name.strip()
        
        if self.machine_number:
            self.machine_number = self.machine_number.strip()
        
        if self.serial_number:
            self.serial_number = self.serial_number.strip()
        
        if self.model_name:
            self.model_name = self.model_name.strip()
        
        if self.manager:
            self.manager = self.manager.strip()
        
        # 제작일 형식 검증 (옵션)
        if self.manufacturing_date:
            self.manufacturing_date = self.manufacturing_date.strip()
            if not self._is_valid_date_format(self.manufacturing_date):
                # 잘못된 형식이어도 원본 데이터 유지 (경고만)
                pass
    
    def _is_valid_date_format(self, date_str: str) -> bool:
        """날짜 형식 검증 (YYYY-MM-DD, YYYY/MM/DD, YYYY.MM.DD 등)"""
        date_patterns = [
            r'^\d{4}-\d{2}-\d{2}$',     # 2023-12-25
            r'^\d{4}/\d{2}/\d{2}$',     # 2023/12/25
            r'^\d{4}\.\d{2}\.\d{2}$',   # 2023.12.25
            r'^\d{4}-\d{1,2}-\d{1,2}$', # 2023-1-1
            r'^\d{4}/\d{1,2}/\d{1,2}$', # 2023/1/1
            r'^\d{4}\.\d{1,2}\.\d{1,2}$' # 2023.1.1
        ]
        
        return any(re.match(pattern, date_str) for pattern in date_patterns)
    
    def to_dict(self) -> Dict[str, Any]:
        """딕셔너리로 변환"""
        return {
            'company_name': self.company_name,
            'machine_number': self.machine_number,
            'serial_number': self.serial_number,
            'manufacturing_date': self.manufacturing_date,
            'model_name': self.model_name,
            'manager': self.manager
        }
    
    @classmethod
    def from_dict(cls, data: Dict[str, Any]) -> 'Equipment':
        """딕셔너리에서 Equipment 객체 생성"""
        return cls(
            company_name=data.get('company_name', ''),
            machine_number=data.get('machine_number'),
            serial_number=data.get('serial_number'),
            manufacturing_date=data.get('manufacturing_date'),
            model_name=data.get('model_name'),
            manager=data.get('manager')
        )
    
    @classmethod
    def from_csv_row(cls, row: Dict[str, str]) -> 'Equipment':
        """CSV 행에서 Equipment 객체 생성 (다양한 컬럼명 지원)"""
        # CSV 컬럼명 매핑 (유연한 컬럼명 지원)
        column_mappings = {
            'company_name': ['회사명', 'company_name', 'company', '업체명', '제조사'],
            'machine_number': ['설비번호', 'machine_number', 'equipment', '장비번호', 'machine_no'],
            'serial_number': ['시리얼넘버', 'serial_number', 'serial', 'sn', '일련번호'],
            'manufacturing_date': ['제작일', 'manufacturing_date', 'mfg_date', '생산일자', '제조일'],
            'model_name': ['모델명', 'model_name', 'model', '모델', 'type'],
            'manager': ['담당자', 'manager', '담당', '관리자']
        }
        
        # 매핑된 값 추출
        mapped_data = {}
        for field, possible_columns in column_mappings.items():
            value = None
            for col in possible_columns:
                if col in row and row[col]:
                    value = row[col].strip() if row[col] else None
                    break
            mapped_data[field] = value
        
        return cls.from_dict(mapped_data)
    
    def is_empty(self) -> bool:
        """빈 설비 정보인지 확인 (회사명 제외한 모든 필드가 비어있음)"""
        return not any([
            self.machine_number,
            self.serial_number,
            self.manufacturing_date,
            self.model_name,
            self.manager
        ])
    
    def has_missing_fields(self) -> list:
        """누락된 필드 목록 반환"""
        missing = []
        
        if not self.machine_number:
            missing.append('설비번호')
        if not self.serial_number:
            missing.append('시리얼넘버')
        if not self.manufacturing_date:
            missing.append('제작일')
        if not self.model_name:
            missing.append('모델명')
        if not self.manager:
            missing.append('담당자')
        
        return missing
    
    def matches_search(self, search_query: str) -> bool:
        """검색어와 일치하는지 확인
        
        검색 방식:
        1. 회사명+설비번호로 검색
        2. 회사명+시리얼넘버로 검색
        """
        if not search_query:
            return False
        
        search_query = search_query.strip().lower()
        
        # 회사명+설비번호 검색
        if self.machine_number:
            company_machine = f"{self.company_name}{self.machine_number}".lower()
            if search_query in company_machine or company_machine in search_query:
                return True
        
        # 회사명+시리얼넘버 검색
        if self.serial_number:
            company_serial = f"{self.company_name}{self.serial_number}".lower()
            if search_query in company_serial or company_serial in search_query:
                return True
        
        # 부분 일치 검색 (회사명, 설비번호, 시리얼넘버 개별)
        searchable_fields = [
            self.company_name,
            self.machine_number,
            self.serial_number,
            self.model_name
        ]
        
        for field in searchable_fields:
            if field and search_query in field.lower():
                return True
        
        return False
    
    def format_display(self) -> str:
        """사용자에게 표시할 형식으로 포맷팅"""
        lines = [f"🏢 회사명: {self.company_name}"]
        
        if self.machine_number:
            lines.append(f"🔧 설비번호: {self.machine_number}")
        else:
            lines.append("🔧 설비번호: [데이터 없음]")
        
        if self.serial_number:
            lines.append(f"🔢 시리얼넘버: {self.serial_number}")
        else:
            lines.append("🔢 시리얼넘버: [데이터 없음]")
        
        if self.manufacturing_date:
            lines.append(f"📅 제작일: {self.manufacturing_date}")
        else:
            lines.append("📅 제작일: [데이터 없음]")
        
        if self.model_name:
            lines.append(f"📋 모델명: {self.model_name}")
        else:
            lines.append("📋 모델명: [데이터 없음]")
        
        if self.manager:
            lines.append(f"👤 담당자: {self.manager}")
        else:
            lines.append("👤 담당자: [데이터 없음]")
        
        return "\n".join(lines)
    
    def __str__(self) -> str:
        """문자열 표현"""
        return f"Equipment(company='{self.company_name}', machine='{self.machine_number}', serial='{self.serial_number}')"
    
    def __repr__(self) -> str:
        """개발자용 문자열 표현"""
        return self.__str__()
    
    def __eq__(self, other) -> bool:
        """설비 객체 동등성 비교 (단순하고 확실한 방법)"""
        if not isinstance(other, Equipment):
            return False
        
        # 회사명, 설비번호, 시리얼넘버 조합으로 고유성 판단 (가장 확실한 방법)
        return (self.company_name == other.company_name and 
                self.machine_number == other.machine_number and
                self.serial_number == other.serial_number)
    
    def __hash__(self) -> int:
        """해시 함수 (__eq__와 일치해야 함)"""
        # __eq__ 메서드와 동일한 필드 조합으로 해시 생성
        return hash((self.company_name, self.machine_number, self.serial_number))


class EquipmentSearchResult:
    """설비 정보 검색 결과"""
    
    def __init__(self, query: str, results: list[Equipment], search_time: float):
        self.query = query
        self.results = results
        self.search_time = search_time
        self.found_count = len(results)
    
    def is_empty(self) -> bool:
        """검색 결과가 비어있는지 확인"""
        return self.found_count == 0
    
    def format_summary(self) -> str:
        """검색 결과 요약 포맷팅"""
        if self.is_empty():
            return f"'{self.query}' 검색 결과: 데이터를 찾을 수 없습니다. (검색 시간: {self.search_time:.3f}초)"
        else:
            return f"'{self.query}' 검색 결과: {self.found_count}건 발견 (검색 시간: {self.search_time:.3f}초)"
    
    def format_detailed(self) -> str:
        """상세 검색 결과 포맷팅"""
        lines = [self.format_summary()]
        
        if not self.is_empty():
            lines.append("")
            for i, equipment in enumerate(self.results, 1):
                lines.append(f"=== 결과 {i} ===")
                lines.append(equipment.format_display())
                lines.append("")
        
        return "\n".join(lines) 