#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
A/S 확인서 데이터 모델

실제 세원 A/S 확인서 엑셀 파일의 셀 매핑을 위한 데이터 구조를 정의합니다.
"""

from dataclasses import dataclass, field
from datetime import datetime
from typing import Optional


@dataclass
class AsReportData:
    """A/S 확인서 생성을 위한 데이터 모델
    
    실제 세원 엑셀 파일의 셀 위치와 정확히 매핑됩니다.
    """
    
    # === 설비 정보 (데이터베이스에서 가져옴) ===
    company_name: str = ""           # G4, G37 - 업체명
    manufacturing_date: str = ""     # AA4 - 제작일
    model_name: str = ""             # N5 - 모델명
    machine_number: str = ""         # N6 - 설비번호
    serial_number: str = ""          # AA5 - 시리얼넘버
    
    # === 사용자 입력 필드 (UI에서 입력) ===
    receipt_date: str = ""           # G8 - 접수일
    as_request: str = ""             # G9 - A/S 신청내역 (선택사항)
    as_repair: str = ""              # G17 - A/S 수리내역 (선택사항)
    check_items: str = ""            # G25 - 체크사항 (선택사항)
    manager: str = ""                # G38 - 담당자 (기본값 DB에서)
    confirmer: str = ""              # X38 - 확인자
    install_type: str = ""           # H7/N7/U7 - 설치구분 (초기/고장수리완료/임시대용)
    
    # === 시스템 자동 입력 ===
    report_year: str = field(default_factory=lambda: datetime.now().strftime("%Y"))    # L36
    report_month: str = field(default_factory=lambda: datetime.now().strftime("%m"))   # P36  
    report_day: str = field(default_factory=lambda: datetime.now().strftime("%d"))     # S36
    
    def get_filename(self) -> str:
        """A/S 확인서 파일명 생성
        
        형식: 업체명_모델명_설비명_AS 확인서_YYMMDD.xlsx
        
        Returns:
            str: 생성된 파일명
        """
        date_str = datetime.now().strftime("%y%m%d")
        
        # 안전한 파일명을 위해 특수문자 제거
        safe_company = self._sanitize_filename(self.company_name)
        safe_model = self._sanitize_filename(self.model_name)
        
        return f"{safe_company}_{safe_model}_AS확인서_{date_str}.xlsx"
    
    def _sanitize_filename(self, text: str) -> str:
        """파일명에 사용할 수 없는 문자 제거
        
        Args:
            text (str): 원본 텍스트
            
        Returns:
            str: 정리된 텍스트
        """
        # Windows에서 파일명에 사용할 수 없는 문자들 제거
        invalid_chars = ['<', '>', ':', '"', '/', '\\', '|', '?', '*']
        
        result = text
        for char in invalid_chars:
            result = result.replace(char, '_')
        
        # 공백을 언더스코어로 변경
        result = result.replace(' ', '_')
        
        # 연속된 언더스코어 정리
        while '__' in result:
            result = result.replace('__', '_')
        
        # 앞뒤 언더스코어 제거
        result = result.strip('_')
        
        return result
    
    def validate_required_fields(self) -> tuple[bool, list[str]]:
        """필수 필드 검증
        
        Returns:
            tuple[bool, list[str]]: (검증 성공 여부, 오류 메시지 목록)
        """
        errors = []
        
        # 필수 필드들 검증
        if not self.company_name.strip():
            errors.append("업체명이 필요합니다")
        
        if not self.model_name.strip():
            errors.append("모델명이 필요합니다")
            
        if not self.receipt_date.strip():
            errors.append("접수일이 필요합니다")
            
        if not self.confirmer.strip():
            errors.append("확인자가 필요합니다")
        
        return len(errors) == 0, errors
    
    def get_cell_mapping(self) -> dict[str, str]:
        """엑셀 셀 매핑 정보 반환
        
        실제 세원 A/S 확인서의 셀 위치와 값들을 매핑합니다.
        
        Returns:
            dict[str, str]: {셀_위치: 값} 형태의 매핑 딕셔너리
        """
        mapping = {
            # 업체명 및 설비 정보
            'G4': self.company_name,          # 업체명 (상단)
            'AA4': self.manufacturing_date,   # 제작일
            'N5': self.model_name,            # 모델명
            'N6': self.machine_number,        # 설비번호  
            'AA5': self.serial_number,        # 시리얼넘버
            
            # 접수 및 작업 내용
            'G8': self.receipt_date,          # 접수일
            'G9': self.as_request,            # A/S 신청내역
            'G17': self.as_repair,            # A/S 수리내역
            'G25': self.check_items,          # 체크사항
            
            # 날짜 정보 (자동)
            'L36': self.report_year,          # 년도 (YYYY)
            'P36': self.report_month,         # 월 (MM)
            'S36': self.report_day,           # 일 (DD)
            
            # 담당자 정보
            'G37': self.company_name,         # 업체명 (하단, G4와 동일)
            'G38': self.manager,              # 담당자
            'X38': self.confirmer,            # 확인자
        }
        
        # 설치구분 체크박스 처리 (H7, N7, U7)
        if self.install_type:
            if self.install_type == "초기":
                mapping['H7'] = "✓"  # 초기(구입) 체크
            elif self.install_type == "고장수리완료":
                mapping['N7'] = "✓"  # 고장수리완료 체크
            elif self.install_type == "임시대용":
                mapping['U7'] = "✓"  # 임시대용 체크
        
        return mapping


@dataclass  
class AsReportSettings:
    """A/S 확인서 시스템 설정"""
    
    # 기본 담당자 (빈 값으로 설정)
    default_manager: str = ""
    
    # 원본 템플릿 파일명
    template_filename: str = "세원_SHT-300VX_1호기_AS 확인서_250708.xlsx"
    
    # 출력 폴더
    output_folder: str = "output" 