#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
설비 정보 보고서 엑셀 템플릿 생성 스크립트

올바른 플레이스홀더 형식으로 템플릿을 생성합니다.
"""

from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill, Alignment, Border, Side
from openpyxl.utils import get_column_letter
from datetime import datetime
import os
from pathlib import Path

def create_template():
    """설비 정보 보고서 템플릿 생성"""
    wb = Workbook()
    ws = wb.active
    ws.title = "설비정보보고서"
    
    # 스타일 정의
    header_font = Font(name='맑은 고딕', size=14, bold=True, color='FFFFFF')
    title_font = Font(name='맑은 고딕', size=16, bold=True, color='2C3E50')
    label_font = Font(name='맑은 고딕', size=11, bold=True, color='34495E')
    data_font = Font(name='맑은 고딕', size=11, color='2C3E50')
    
    header_fill = PatternFill(start_color='3498DB', end_color='3498DB', fill_type='solid')
    label_fill = PatternFill(start_color='ECF0F1', end_color='ECF0F1', fill_type='solid')
    
    center_align = Alignment(horizontal='center', vertical='center')
    left_align = Alignment(horizontal='left', vertical='center')
    
    thin_border = Border(
        left=Side(style='thin'),
        right=Side(style='thin'),
        top=Side(style='thin'),
        bottom=Side(style='thin')
    )
    
    # 제목
    ws['A1'] = '설비 정보 보고서'
    ws['A1'].font = title_font
    ws['A1'].alignment = center_align
    ws.merge_cells('A1:F1')
    
    # 생성일시
    ws['A2'] = f'생성일시: {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}'
    ws['A2'].font = data_font
    ws['A2'].alignment = left_align
    ws.merge_cells('A2:F2')
    
    # 헤더 행
    ws['A4'] = '항목'
    ws['B4'] = '내용'
    ws['C4'] = '비고'
    
    for col in ['A4', 'B4', 'C4']:
        ws[col].font = header_font
        ws[col].fill = header_fill
        ws[col].alignment = center_align
        ws[col].border = thin_border
    
    # 데이터 필드들 (올바른 플레이스홀더 형식)
    fields = [
        ('회사명', 'COMPANY_NAME'),
        ('설비명', 'EQUIPMENT_NAME'),
        ('시리얼넘버', 'SERIAL_NUMBER'),
        ('제작일', 'MANUFACTURING_DATE'),
        ('모델명', 'MODEL_NAME'),
        ('스핀들시리얼넘버', 'SPINDLE_SERIAL_NUMBER')
    ]
    
    row = 5
    for field_name, field_code in fields:
        # 항목명
        ws[f'A{row}'] = field_name
        ws[f'A{row}'].font = label_font
        ws[f'A{row}'].fill = label_fill
        ws[f'A{row}'].alignment = center_align
        ws[f'A{row}'].border = thin_border
        
        # 내용 (올바른 매핑 필드 - 중괄호 하나씩만)
        ws[f'B{row}'] = f'{{{field_code}}}'  # {COMPANY_NAME} 형식
        ws[f'B{row}'].font = data_font
        ws[f'B{row}'].alignment = left_align
        ws[f'B{row}'].border = thin_border
        
        # 비고
        ws[f'C{row}'] = ''
        ws[f'C{row}'].font = data_font
        ws[f'C{row}'].alignment = left_align
        ws[f'C{row}'].border = thin_border
        
        row += 1
    
    # 추가 정보 섹션
    row += 1
    ws[f'A{row}'] = '추가 정보'
    ws[f'A{row}'].font = title_font
    ws[f'A{row}'].alignment = left_align
    ws.merge_cells(f'A{row}:C{row}')
    
    row += 1
    ws[f'A{row}'] = '생성자'
    ws[f'A{row}'].font = label_font
    ws[f'A{row}'].fill = label_fill
    ws[f'A{row}'].alignment = center_align
    ws[f'A{row}'].border = thin_border
    
    ws[f'B{row}'] = 'N1VI 설비 정보 시스템'
    ws[f'B{row}'].font = data_font
    ws[f'B{row}'].alignment = left_align
    ws[f'B{row}'].border = thin_border
    
    ws[f'C{row}'] = 'v1.0'
    ws[f'C{row}'].font = data_font
    ws[f'C{row}'].alignment = center_align
    ws[f'C{row}'].border = thin_border
    
    # 컬럼 너비 조정
    ws.column_dimensions['A'].width = 20
    ws.column_dimensions['B'].width = 30
    ws.column_dimensions['C'].width = 15
    
    # 행 높이 조정
    for i in range(1, row + 1):
        ws.row_dimensions[i].height = 25
    
    return wb

def main():
    """메인 함수"""
    print("🔧 엑셀 템플릿 생성 중...")
    
    # 프로젝트 루트 찾기
    project_root = Path(__file__).parent.parent
    template_dir = project_root / "templates"
    
    # 템플릿 디렉토리 생성
    template_dir.mkdir(exist_ok=True)
    
    # 템플릿 생성
    wb = create_template()
    template_path = template_dir / 'equipment_report_template.xlsx'
    
    # 기존 파일이 있으면 백업
    if template_path.exists():
        backup_path = template_dir / f'equipment_report_template_backup_{datetime.now().strftime("%Y%m%d_%H%M%S")}.xlsx'
        template_path.rename(backup_path)
        print(f"📁 기존 템플릿 백업: {backup_path}")
    
    wb.save(template_path)
    wb.close()
    
    print(f"✅ 엑셀 템플릿이 성공적으로 생성되었습니다!")
    print(f"   경로: {template_path}")
    print(f"   크기: {template_path.stat().st_size:,} bytes")

if __name__ == "__main__":
    main() 