#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
실제 업무용 A/S 확인서 형태의 엑셀 템플릿 생성 스크립트

세원 A/S 확인서를 기반으로 설비 정보 매핑이 가능한 템플릿을 생성합니다.
"""

from openpyxl import Workbook
from openpyxl.styles import Font, PatternFill, Alignment, Border, Side
from openpyxl.utils import get_column_letter
from datetime import datetime
import os
from pathlib import Path

def create_business_template():
    """실제 업무용 A/S 확인서 템플릿 생성"""
    wb = Workbook()
    ws = wb.active
    ws.title = "설비정보확인서"
    
    # 스타일 정의
    title_font = Font(name='맑은 고딕', size=18, bold=True, color='000000')
    header_font = Font(name='맑은 고딕', size=12, bold=True, color='000000')
    label_font = Font(name='맑은 고딕', size=11, bold=True, color='000000')
    data_font = Font(name='맑은 고딕', size=11, color='000000')
    
    # 정렬
    center_align = Alignment(horizontal='center', vertical='center')
    left_align = Alignment(horizontal='left', vertical='center')
    right_align = Alignment(horizontal='right', vertical='center')
    
    # 테두리
    thin_border = Border(
        left=Side(style='thin'),
        right=Side(style='thin'),
        top=Side(style='thin'),
        bottom=Side(style='thin')
    )
    
    # 배경색
    light_blue = PatternFill(start_color='E3F2FD', end_color='E3F2FD', fill_type='solid')
    light_gray = PatternFill(start_color='F5F5F5', end_color='F5F5F5', fill_type='solid')
    
    # 컬럼 너비 설정
    column_widths = {
        'A': 15, 'B': 8, 'C': 8, 'D': 8, 'E': 8, 'F': 8,
        'G': 20, 'H': 8, 'I': 15, 'J': 8, 'K': 15
    }
    
    for col, width in column_widths.items():
        ws.column_dimensions[col].width = width
    
    # 제목 (A1:K1 병합)
    ws['A1'] = '설비 정보 확인서'
    ws['A1'].font = title_font
    ws['A1'].alignment = center_align
    ws.merge_cells('A1:K1')
    ws.row_dimensions[1].height = 30
    
    # 빈 행
    ws.row_dimensions[2].height = 15
    ws.row_dimensions[3].height = 15
    
    # 업체명 (4행)
    ws['A4'] = '업체명'
    ws['A4'].font = label_font
    ws['A4'].alignment = center_align
    ws['A4'].fill = light_gray
    ws['A4'].border = thin_border
    
    ws['G4'] = '{COMPANY_NAME}'
    ws['G4'].font = data_font
    ws['G4'].alignment = left_align
    ws['G4'].border = thin_border
    ws.merge_cells('G4:K4')
    
    # 장비 내용 (5-6행)
    ws['A5'] = '장 비  내 용'
    ws['A5'].font = label_font
    ws['A5'].alignment = center_align
    ws['A5'].fill = light_gray
    ws['A5'].border = thin_border
    ws.merge_cells('A5:A6')
    
    ws['G5'] = '장 비 명'
    ws['G5'].font = label_font
    ws['G5'].alignment = center_align
    ws['G5'].fill = light_gray
    ws['G5'].border = thin_border
    ws.merge_cells('G5:K5')
    
    ws['G6'] = '{EQUIPMENT_NAME}'
    ws['G6'].font = data_font
    ws['G6'].alignment = left_align
    ws['G6'].border = thin_border
    ws.merge_cells('G6:K6')
    
    # 호기 정보 (7행)
    ws['G7'] = '호        기'
    ws['G7'].font = label_font
    ws['G7'].alignment = center_align
    ws['G7'].fill = light_gray
    ws['G7'].border = thin_border
    ws.merge_cells('G7:I7')
    
    ws['J7'] = '{SERIAL_NUMBER}'
    ws['J7'].font = data_font
    ws['J7'].alignment = center_align
    ws['J7'].border = thin_border
    ws.merge_cells('J7:K7')
    
    # 설치 구분 (8행)
    ws['A8'] = '설 치  구 분'
    ws['A8'].font = label_font
    ws['A8'].alignment = center_align
    ws['A8'].fill = light_gray
    ws['A8'].border = thin_border
    ws.merge_cells('A8:F8')
    
    ws['I8'] = '초기(구입)'
    ws['I8'].font = data_font
    ws['I8'].alignment = center_align
    ws['I8'].border = thin_border
    ws.merge_cells('I8:K8')
    
    # 접수 일자 (9행)
    ws['A9'] = '접수 일자'
    ws['A9'].font = label_font
    ws['A9'].alignment = center_align
    ws['A9'].fill = light_gray
    ws['A9'].border = thin_border
    ws.merge_cells('A9:F9')
    
    ws['G9'] = '{MANUFACTURING_DATE}'
    ws['G9'].font = data_font
    ws['G9'].alignment = center_align
    ws['G9'].border = thin_border
    ws.merge_cells('G9:K9')
    
    # 설비 신청내역 (10-16행)
    ws['A10'] = '설비 정보 내역'
    ws['A10'].font = label_font
    ws['A10'].alignment = center_align
    ws['A10'].fill = light_gray
    ws['A10'].border = thin_border
    ws.merge_cells('A10:F10')
    
    # 내역 입력 영역
    equipment_info = [
        '1. 모델명: {MODEL_NAME}',
        '',
        '2. 스핀들 시리얼넘버: {SPINDLE_SERIAL_NUMBER}',
        '',
        '3. 기타 정보:',
        '   - 생성일시: ' + datetime.now().strftime("%Y-%m-%d %H:%M:%S"),
        '   - 생성시스템: N1VI 설비 정보 시스템 v1.0'
    ]
    
    for i, info in enumerate(equipment_info, start=10):
        if i <= 16:
            ws[f'G{i}'] = info
            ws[f'G{i}'].font = data_font
            ws[f'G{i}'].alignment = left_align
            ws[f'G{i}'].border = thin_border
            ws.merge_cells(f'G{i}:K{i}')
    
    # 확인 내역 (17-30행)
    ws['A17'] = '설비 확인내역'
    ws['A17'].font = label_font
    ws['A17'].alignment = center_align
    ws['A17'].fill = light_blue
    ws['A17'].border = thin_border
    ws.merge_cells('A17:F17')
    
    # 확인 내역 입력 영역 (빈 공간으로 유지)
    for row in range(17, 31):
        ws[f'G{row}'] = ''
        ws[f'G{row}'].border = thin_border
        ws.merge_cells(f'G{row}:K{row}')
    
    # 하단 서명란 (32-35행)
    ws['A32'] = '확인 사항'
    ws['A32'].font = label_font
    ws['A32'].alignment = center_align
    ws['A32'].fill = light_gray
    ws['A32'].border = thin_border
    ws.merge_cells('A32:K32')
    
    ws['A33'] = '위 내용으로 설비 정보를 확인합니다.'
    ws['A33'].font = data_font
    ws['A33'].alignment = center_align
    ws.merge_cells('A33:K33')
    
    # 날짜 및 서명
    ws['A35'] = f"확인일자: {datetime.now().strftime('%Y년 %m월 %d일')}"
    ws['A35'].font = data_font
    ws['A35'].alignment = left_align
    ws.merge_cells('A35:E35')
    
    ws['G35'] = '확인자: __________________ (인)'
    ws['G35'].font = data_font
    ws['G35'].alignment = right_align
    ws.merge_cells('G35:K35')
    
    # 행 높이 설정
    for row in range(4, 36):
        if row in [4, 8, 9, 10, 17, 32]:
            ws.row_dimensions[row].height = 25
        else:
            ws.row_dimensions[row].height = 20
    
    return wb

def main():
    """메인 함수"""
    print("🔧 실제 업무용 설비 정보 확인서 템플릿 생성 중...")
    
    # 프로젝트 루트 찾기
    project_root = Path(__file__).parent.parent
    template_dir = project_root / "templates"
    
    # 템플릿 디렉토리 생성
    template_dir.mkdir(exist_ok=True)
    
    # 템플릿 생성
    wb = create_business_template()
    template_path = template_dir / 'business_equipment_template.xlsx'
    
    # 기존 파일이 있으면 백업
    if template_path.exists():
        backup_path = template_dir / f'business_equipment_template_backup_{datetime.now().strftime("%Y%m%d_%H%M%S")}.xlsx'
        template_path.rename(backup_path)
        print(f"📁 기존 템플릿 백업: {backup_path}")
    
    wb.save(template_path)
    wb.close()
    
    print(f"✅ 실제 업무용 설비 정보 확인서 템플릿이 성공적으로 생성되었습니다!")
    print(f"   경로: {template_path}")
    print(f"   크기: {template_path.stat().st_size:,} bytes")

if __name__ == "__main__":
    main() 