#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
세원 A/S 확인서 상세 분석 스크립트

실제 업무용 엑셀 파일의 셀 구조를 분석하여 매핑 정보를 확인합니다.
"""

from openpyxl import load_workbook
from pathlib import Path
import sys

def analyze_sewon_excel():
    """세원 A/S 확인서 분석"""
    project_root = Path(__file__).parent.parent
    excel_file = project_root / "세원_SHT-300VX_1호기_AS 확인서_250708.xlsx"
    
    if not excel_file.exists():
        print(f"❌ 파일을 찾을 수 없습니다: {excel_file}")
        return False
    
    print(f"📊 세원 A/S 확인서 분석: {excel_file}")
    print("=" * 80)
    
    try:
        workbook = load_workbook(excel_file, data_only=True)
        worksheet = workbook.active
        
        print(f"📋 워크시트: '{worksheet.title}'")
        print(f"📏 크기: {worksheet.max_row}행 x {worksheet.max_column}열")
        print()
        
        # 매핑해야 할 셀들 확인
        mapping_cells = {
            'G4': '업체명',
            'AA4': '제작일', 
            'N5': '모델명',
            'AA5': '시리얼넘버',
            'G8': '접수일',
            'G9': 'A/S 신청내역',
            'G17': 'A/S 수리내역', 
            'G25': '체크사항',
            'L36': '년도(YYYY)',
            'P36': '월(MM)',
            'S36': '일(DD)',
            'G37': '업체명(하단)',
            'G38': '담당자',
            'X38': '확인자'
        }
        
        print("🎯 매핑 대상 셀 분석:")
        print("-" * 60)
        
        for cell_addr, description in mapping_cells.items():
            try:
                cell = worksheet[cell_addr]
                value = cell.value
                
                print(f"  {cell_addr:>4} ({description:>12}): {value}")
                
                # 셀 스타일 정보
                if hasattr(cell, 'font') and cell.font:
                    font_info = f"폰트: {cell.font.name or 'default'}, 크기: {cell.font.size or 'default'}"
                    if cell.font.bold:
                        font_info += ", 굵게"
                    print(f"        스타일: {font_info}")
                
                # 병합된 셀 확인
                for merged_range in worksheet.merged_cells.ranges:
                    if cell_addr in merged_range:
                        print(f"        병합: {merged_range}")
                        break
                
            except Exception as e:
                print(f"  {cell_addr:>4} ({description:>12}): ❌ 오류 - {e}")
        
        print()
        print("🔍 주변 셀 컨텍스트 분석:")
        print("-" * 60)
        
        # G4 주변 (업체명)
        print("  G4 주변 (업체명):")
        for row in range(3, 6):
            for col in range(6, 9):  # F, G, H
                cell_addr = f"{chr(65+col-1)}{row}"
                try:
                    value = worksheet[cell_addr].value
                    print(f"    {cell_addr}: {value}")
                except:
                    print(f"    {cell_addr}: [오류]")
        
        print()
        
        # AA4, AA5 주변 (제작일, 시리얼넘버)
        print("  AA4, AA5 주변 (제작일, 시리얼넘버):")
        aa_cells = ['Z4', 'AA4', 'AB4', 'Z5', 'AA5', 'AB5', 'Z6', 'AA6', 'AB6']
        for cell_addr in aa_cells:
            try:
                value = worksheet[cell_addr].value
                print(f"    {cell_addr}: {value}")
            except:
                print(f"    {cell_addr}: [오류]")
        
        workbook.close()
        print("\n✅ 분석 완료!")
        return True
        
    except Exception as e:
        print(f"❌ 분석 실패: {e}")
        return False

def main():
    """메인 함수"""
    analyze_sewon_excel()

if __name__ == "__main__":
    main() 