#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
실제 업무용 엑셀 파일 구조 분석 스크립트

세원_SHT-300VX_1호기_AS 확인서_250708.xlsx 파일을 분석하여
새로운 템플릿 생성을 위한 정보를 수집합니다.
"""

from openpyxl import load_workbook
from pathlib import Path
import sys

def analyze_excel_file(file_path):
    """엑셀 파일 분석"""
    print(f"📊 엑셀 파일 분석 시작: {file_path}")
    print("=" * 80)
    
    try:
        # 엑셀 파일 로드
        wb = load_workbook(file_path, data_only=True)
        
        print(f"📋 워크시트 목록:")
        for i, sheet_name in enumerate(wb.sheetnames, 1):
            print(f"   {i}. {sheet_name}")
        print()
        
        # 각 워크시트 분석
        for sheet_name in wb.sheetnames:
            print(f"📄 워크시트: '{sheet_name}' 분석")
            print("-" * 60)
            
            ws = wb[sheet_name]
            
            # 시트 크기 정보
            print(f"   크기: {ws.max_row}행 x {ws.max_column}열")
            print(f"   사용 범위: {ws.calculate_dimension()}")
            print()
            
            # 첫 20행의 내용 출력
            print("   첫 20행 내용:")
            for row_num in range(1, min(21, ws.max_row + 1)):
                row_data = []
                for col_num in range(1, min(11, ws.max_column + 1)):  # 최대 10열까지
                    cell = ws.cell(row=row_num, column=col_num)
                    value = cell.value
                    if value is not None:
                        # 긴 텍스트는 줄임
                        value_str = str(value)
                        if len(value_str) > 30:
                            value_str = value_str[:27] + "..."
                        row_data.append(f"[{value_str}]")
                    else:
                        row_data.append("[빈칸]")
                
                print(f"   {row_num:2d}: {' | '.join(row_data)}")
            
            if ws.max_row > 20:
                print(f"   ... (총 {ws.max_row}행 중 20행만 표시)")
            
            print()
            print()
        
        # 특정 패턴 찾기
        print("🔍 패턴 분석:")
        print("-" * 60)
        
        # 모든 시트에서 특정 키워드 찾기
        keywords = ['회사', '설비', '시리얼', '모델', '제작', '번호', '명칭', '날짜', '확인']
        
        for sheet_name in wb.sheetnames:
            ws = wb[sheet_name]
            print(f"   '{sheet_name}' 시트에서 키워드 검색:")
            
            for row_num in range(1, min(50, ws.max_row + 1)):
                for col_num in range(1, min(20, ws.max_column + 1)):
                    cell = ws.cell(row=row_num, column=col_num)
                    if cell.value and isinstance(cell.value, str):
                        for keyword in keywords:
                            if keyword in cell.value:
                                print(f"      {keyword} 발견: ({row_num}, {col_num}) = '{cell.value}'")
            print()
        
        wb.close()
        print("✅ 분석 완료!")
        
    except Exception as e:
        print(f"❌ 분석 실패: {e}")
        return False
    
    return True

def main():
    """메인 함수"""
    project_root = Path(__file__).parent.parent
    excel_file = project_root / "세원_SHT-300VX_1호기_AS 확인서_250708.xlsx"
    
    if not excel_file.exists():
        print(f"❌ 파일을 찾을 수 없습니다: {excel_file}")
        sys.exit(1)
    
    analyze_excel_file(excel_file)

if __name__ == "__main__":
    main() 