#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
N1VI 애플리케이션 실행 스크립트 (개발용)

개발 환경에서 애플리케이션을 실행하기 위한 스크립트입니다.
"""

import sys
import os
from pathlib import Path

def check_dependencies():
    """의존성 검사"""
    print("🔍 의존성 검사 중...")
    
    required_packages = [
        'PySide6',
        'pandas',
        'openpyxl',
        'requests',
        'tqdm'
    ]
    
    missing_packages = []
    
    for package in required_packages:
        try:
            __import__(package)
            print(f"  ✅ {package}")
        except ImportError:
            missing_packages.append(package)
            print(f"  ❌ {package} - 누락됨")
    
    if missing_packages:
        print(f"\n❌ 누락된 패키지: {', '.join(missing_packages)}")
        print("다음 명령어로 설치하세요:")
        print("pip install -r requirements.txt")
        return False
    
    print("✅ 모든 의존성 확인 완료!")
    return True

def setup_environment():
    """개발 환경 설정"""
    print("⚙️  개발 환경 설정 중...")
    
    # 프로젝트 루트를 Python 경로에 추가
    project_root = Path(__file__).parent
    src_path = project_root / "src"
    
    if str(src_path) not in sys.path:
        sys.path.insert(0, str(src_path))
    
    # 환경 변수 설정
    os.environ['PYTHONPATH'] = str(src_path)
    os.environ['N1VI_DEV_MODE'] = 'true'
    
    print("✅ 환경 설정 완료!")

def run_application():
    """애플리케이션 실행"""
    print("🚀 N1VI 애플리케이션 실행 중...")
    
    try:
        # main.py 실행
        import main
        return main.main()
    except Exception as e:
        print(f"❌ 애플리케이션 실행 실패: {e}")
        import traceback
        traceback.print_exc()
        return 1

def main():
    """메인 함수"""
    print("=" * 50)
    print("🎯 N1VI 개발 환경 실행기")
    print("=" * 50)
    
    # 1. 의존성 검사
    if not check_dependencies():
        return 1
    
    # 2. 환경 설정
    setup_environment()
    
    # 3. 애플리케이션 실행
    exit_code = run_application()
    
    print("=" * 50)
    if exit_code == 0:
        print("✅ 애플리케이션이 정상적으로 종료되었습니다.")
    else:
        print("❌ 애플리케이션 실행 중 오류가 발생했습니다.")
    
    return exit_code

if __name__ == "__main__":
    sys.exit(main()) 