#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
N1VI 설비 정보 검색 & A/S 확인서 생성 시스템 메인 실행 파일

실제 세원 A/S 확인서를 기반으로 한 설비 관리 시스템입니다.
"""

import logging
import sys
from pathlib import Path

# PySide6 설정
from PySide6.QtWidgets import QApplication
from PySide6.QtCore import Qt

# 프로젝트 루트를 Python 경로에 추가
project_root = Path(__file__).parent
if str(project_root) not in sys.path:
    sys.path.insert(0, str(project_root))

# N1VI 모듈 임포트
from src.n1vi.ui.main_window import MainWindow
from src.n1vi.services.equipment_service import EquipmentService
from src.n1vi.services.as_service import AsService


def setup_logging():
    """로깅 설정"""
    logging.basicConfig(
        level=logging.INFO,
        format='%(asctime)s - %(name)s - %(levelname)s - %(message)s',
        handlers=[
            logging.StreamHandler(sys.stdout)
        ]
    )


def check_data_file():
    """데이터 파일 존재 확인"""
    data_file = project_root / "data" / "equipment_data.csv"
    
    if data_file.exists():
        print(f"✅ 설비 데이터 파일 발견: {data_file}")
        return True
    else:
        print(f"❌ 설비 데이터 파일 없음: {data_file}")
        print("   data/equipment_data.csv 파일이 필요합니다.")
        return False


def check_template_file():
    """템플릿 파일 존재 확인"""
    template_file = project_root / "세원_SHT-300VX_1호기_AS 확인서_250708.xlsx"
    
    if template_file.exists():
        print(f"✅ A/S 확인서 템플릿 발견: {template_file}")
        return True
    else:
        print(f"❌ A/S 확인서 템플릿 없음: {template_file}")
        print("   세원_SHT-300VX_1호기_AS 확인서_250708.xlsx 파일이 필요합니다.")
        return False


def create_output_directory():
    """출력 디렉토리 생성"""
    output_dir = project_root / "output"
    output_dir.mkdir(exist_ok=True)
    print(f"✅ 출력 폴더 준비: {output_dir}")


def main():
    """메인 함수"""
    print("🔧 N1VI 설비 정보 검색 & A/S 확인서 생성 시스템")
    print("=" * 70)
    
    # 로깅 설정
    setup_logging()
    logger = logging.getLogger(__name__)
    
    try:
        # 파일 시스템 검증
        if not check_data_file():
            input("Enter 키를 누르면 종료됩니다...")
            return 1
        
        if not check_template_file():
            input("Enter 키를 누르면 종료됩니다...")
            return 1
        
        # 출력 디렉토리 생성
        create_output_directory()
        
        print("🚀 A/S 확인서 생성 GUI를 시작합니다...")
        
        # Qt 애플리케이션 초기화
        app = QApplication(sys.argv)
        app.setApplicationName("N1VI A/S 확인서 시스템")
        app.setApplicationVersion("2.0.0")
        app.setOrganizationName("N1VI")
        
        # 고해상도 디스플레이 지원
        app.setAttribute(Qt.AA_EnableHighDpiScaling)
        app.setAttribute(Qt.AA_UseHighDpiPixmaps)
        
        # 메인 윈도우 생성 및 표시
        window = MainWindow()
        window.show()
        
        logger.info("N1VI A/S 확인서 시스템 시작 완료")
        
        # 애플리케이션 실행
        return app.exec()
        
    except Exception as e:
        logger.error(f"시스템 시작 실패: {e}", exc_info=True)
        print(f"\n❌ 시스템 시작 중 오류 발생: {e}")
        input("Enter 키를 누르면 종료됩니다...")
        return 1


if __name__ == "__main__":
    sys.exit(main()) 