#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
N1VI 애플리케이션 설정

전역 설정 및 환경 변수를 관리합니다.
"""

import os
import json
from pathlib import Path
from typing import Dict, Any

class Config:
    """애플리케이션 설정 클래스"""
    
    def __init__(self):
        self.config_file = Path("n1vi_config.json")
        self.default_config = self.get_default_config()
        self.config = self.load_config()
    
    def get_default_config(self) -> Dict[str, Any]:
        """기본 설정 반환"""
        return {
            "app": {
                "name": "N1VI",
                "version": "1.0.0",
                "language": "ko",
                "theme": "default"
            },
            "database": {
                "cache_file": "data/n1vi_cache.sqlite",
                "csv_file": "data/company_equipment.csv",
                "backup_enabled": True,
                "backup_interval": 24  # 시간 단위
            },
            "api": {
                "csv_url": "https://example.com/api/equipment.csv",
                "timeout": 30,
                "retry_count": 3,
                "retry_delay": 1
            },
            "excel": {
                "template_dir": "templates",
                "output_dir": "output",
                "default_template": "default_report.xlsx",
                "auto_open": True
            },
            "ui": {
                "window_width": 800,
                "window_height": 600,
                "auto_complete": True,
                "max_recent_items": 10
            },
            "logging": {
                "level": "INFO",
                "file": "logs/n1vi.log",
                "max_size": 10485760,  # 10MB
                "backup_count": 5
            },
            "security": {
                "encrypt_cache": True,
                "verify_ssl": True,
                "auto_update": True
            }
        }
    
    def load_config(self) -> Dict[str, Any]:
        """설정 파일 로드"""
        if self.config_file.exists():
            try:
                with open(self.config_file, 'r', encoding='utf-8') as f:
                    loaded_config = json.load(f)
                    # 기본 설정과 병합
                    config = self.default_config.copy()
                    config.update(loaded_config)
                    return config
            except Exception as e:
                print(f"설정 파일 로드 실패: {e}")
                return self.default_config
        else:
            # 기본 설정으로 파일 생성
            self.save_config(self.default_config)
            return self.default_config
    
    def save_config(self, config: Dict[str, Any] = None):
        """설정 파일 저장"""
        if config is None:
            config = self.config
        
        try:
            with open(self.config_file, 'w', encoding='utf-8') as f:
                json.dump(config, f, ensure_ascii=False, indent=2)
        except Exception as e:
            print(f"설정 파일 저장 실패: {e}")
    
    def get(self, key: str, default=None):
        """설정값 조회 (점 표기법 지원)"""
        keys = key.split('.')
        value = self.config
        
        for k in keys:
            if isinstance(value, dict) and k in value:
                value = value[k]
            else:
                return default
                
        return value
    
    def set(self, key: str, value: Any):
        """설정값 변경 (점 표기법 지원)"""
        keys = key.split('.')
        config = self.config
        
        for k in keys[:-1]:
            if k not in config:
                config[k] = {}
            config = config[k]
        
        config[keys[-1]] = value
        self.save_config()
    
    def reset_to_default(self):
        """기본 설정으로 초기화"""
        self.config = self.default_config.copy()
        self.save_config()

# 글로벌 설정 인스턴스
config = Config()

# 편의 함수들
def get_config(key: str, default=None):
    """설정값 조회"""
    return config.get(key, default)

def set_config(key: str, value: Any):
    """설정값 변경"""
    config.set(key, value)

def create_directories():
    """필요한 디렉토리 생성"""
    directories = [
        "data",
        "templates", 
        "output",
        "logs",
        "backup"
    ]
    
    for directory in directories:
        Path(directory).mkdir(exist_ok=True)

# 초기화 시 디렉토리 생성
create_directories()

# 환경 변수 처리
def get_env_config():
    """환경 변수에서 설정 로드"""
    env_config = {}
    
    # 환경 변수 매핑
    env_mappings = {
        "N1VI_CSV_URL": "api.csv_url",
        "N1VI_LOG_LEVEL": "logging.level",
        "N1VI_ENCRYPT_CACHE": "security.encrypt_cache",
        "N1VI_AUTO_UPDATE": "security.auto_update"
    }
    
    for env_var, config_key in env_mappings.items():
        value = os.getenv(env_var)
        if value is not None:
            # 타입 변환
            if value.lower() in ('true', 'false'):
                value = value.lower() == 'true'
            elif value.isdigit():
                value = int(value)
            
            set_config(config_key, value)
    
    return env_config

# 환경 변수 설정 적용
get_env_config()

if __name__ == "__main__":
    # 설정 테스트
    print("=== N1VI 설정 테스트 ===")
    print(f"앱 이름: {get_config('app.name')}")
    print(f"앱 버전: {get_config('app.version')}")
    print(f"데이터베이스 파일: {get_config('database.cache_file')}")
    print(f"로그 레벨: {get_config('logging.level')}")
    print(f"설정 파일: {config.config_file}")
    print("테스트 완료!") 