#!/usr/bin/env python3
# -*- coding: utf-8 -*-
"""
N1VI 애플리케이션 빌드 스크립트

PyInstaller를 사용하여 실행 파일을 생성합니다.
"""

import os
import sys
import shutil
import subprocess
from pathlib import Path

def clean_build():
    """빌드 디렉토리 정리"""
    print("🧹 빌드 디렉토리 정리 중...")
    
    dirs_to_clean = ['build', 'dist', '__pycache__']
    
    for dir_name in dirs_to_clean:
        if Path(dir_name).exists():
            shutil.rmtree(dir_name)
            print(f"  ✅ {dir_name} 삭제됨")
    
    # .pyc 파일 정리
    for root, dirs, files in os.walk('.'):
        for file in files:
            if file.endswith('.pyc'):
                os.remove(os.path.join(root, file))
    
    print("✅ 정리 완료!")

def install_dependencies():
    """의존성 설치"""
    print("📦 의존성 설치 중...")
    
    try:
        subprocess.check_call([sys.executable, "-m", "pip", "install", "-r", "requirements.txt"])
        print("✅ 의존성 설치 완료!")
    except subprocess.CalledProcessError as e:
        print(f"❌ 의존성 설치 실패: {e}")
        return False
    
    return True

def build_executable():
    """실행 파일 빌드"""
    print("🔨 실행 파일 빌드 중...")
    
    # PyInstaller 옵션
    options = [
        "--onefile",  # 단일 파일로 빌드
        "--windowed",  # 콘솔 창 숨김 (Windows)
        "--name=N1VI",  # 실행 파일 이름
        "--icon=assets/icon.ico" if Path("assets/icon.ico").exists() else "",  # 아이콘 (옵션)
        "--add-data=templates:templates",  # 템플릿 폴더 포함
        "--add-data=data:data",  # 데이터 폴더 포함
        "--hidden-import=PySide6.QtCore",
        "--hidden-import=PySide6.QtGui",
        "--hidden-import=PySide6.QtWidgets",
        "--hidden-import=pandas",
        "--hidden-import=openpyxl",
        "--hidden-import=requests",
        "--hidden-import=tqdm",
        "--hidden-import=sqlite3",
        "--hidden-import=json",
        "--hidden-import=pathlib",
        "--hidden-import=logging",
        "--clean",  # 빌드 전 정리
        "main.py"
    ]
    
    # 빈 옵션 제거
    options = [opt for opt in options if opt]
    
    try:
        subprocess.check_call([sys.executable, "-m", "PyInstaller"] + options)
        print("✅ 빌드 완료!")
        return True
    except subprocess.CalledProcessError as e:
        print(f"❌ 빌드 실패: {e}")
        return False

def create_installer():
    """설치 프로그램 생성 (옵션)"""
    print("📦 설치 프로그램 생성 중...")
    
    # 여기에 NSIS 또는 다른 설치 프로그램 생성 로직을 추가할 수 있습니다.
    # 현재는 간단히 배포 폴더 생성
    
    dist_dir = Path("dist")
    release_dir = Path("release")
    
    if release_dir.exists():
        shutil.rmtree(release_dir)
    
    release_dir.mkdir()
    
    # 실행 파일 복사
    exe_file = dist_dir / "N1VI.exe"
    if exe_file.exists():
        shutil.copy2(exe_file, release_dir)
    
    # 설정 파일 및 문서 복사
    files_to_copy = [
        "README.md",
        "requirements.txt",
        "n1vi_config.json"
    ]
    
    for file_name in files_to_copy:
        if Path(file_name).exists():
            shutil.copy2(file_name, release_dir)
    
    # 템플릿 및 데이터 폴더 복사
    dirs_to_copy = ["templates", "data"]
    for dir_name in dirs_to_copy:
        if Path(dir_name).exists():
            shutil.copytree(dir_name, release_dir / dir_name, dirs_exist_ok=True)
    
    print("✅ 배포 패키지 생성 완료!")
    print(f"📁 배포 폴더: {release_dir.absolute()}")

def test_build():
    """빌드된 실행 파일 테스트"""
    print("🧪 빌드 테스트 중...")
    
    exe_path = Path("dist/N1VI.exe")
    if not exe_path.exists():
        print("❌ 실행 파일을 찾을 수 없습니다.")
        return False
    
    print(f"✅ 실행 파일 생성됨: {exe_path.absolute()}")
    print(f"📏 파일 크기: {exe_path.stat().st_size / 1024 / 1024:.2f} MB")
    
    # 여기에 실행 파일 테스트 로직을 추가할 수 있습니다.
    return True

def main():
    """메인 빌드 함수"""
    print("🚀 N1VI 애플리케이션 빌드 시작")
    print("=" * 50)
    
    # 1. 빌드 디렉토리 정리
    clean_build()
    
    # 2. 의존성 설치
    if not install_dependencies():
        return 1
    
    # 3. 실행 파일 빌드
    if not build_executable():
        return 1
    
    # 4. 빌드 테스트
    if not test_build():
        return 1
    
    # 5. 설치 프로그램 생성
    create_installer()
    
    print("=" * 50)
    print("🎉 빌드 완료!")
    print("📁 실행 파일: dist/N1VI.exe")
    print("📦 배포 패키지: release/")
    
    return 0

if __name__ == "__main__":
    sys.exit(main()) 